<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  users.php — User Account Management
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - List all user accounts (accountants & clients)
 *    - Create new user accounts
 *    - Edit existing accounts (name, email, role, client link)
 *    - Activate / Deactivate accounts
 *    - Delete accounts (with protection for self-deletion)
 *    - Link client portal users to their client record
 *
 *  Authorization: Accountant only
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

$current_user_id = (int) $_SESSION['user_id'];

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST ACTIONS
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('users.php');
    }

    $action = $_POST['action'] ?? '';

    /* ── DELETE ─────────────────────────────────────────────── */
    if ($action === 'delete') {
        $del_id = (int) ($_POST['user_id'] ?? 0);

        if ($del_id === $current_user_id) {
            flash('error', 'You cannot delete your own account.');
        } elseif ($del_id > 0) {
            db_execute("DELETE FROM users WHERE id = ?", ['i', $del_id]);
            flash('success', 'User account deleted.');
        }
        csrf_regenerate();
        redirect('users.php');
    }

    /* ── TOGGLE ACTIVE ─────────────────────────────────────── */
    if ($action === 'toggle_active') {
        $toggle_id = (int) ($_POST['user_id'] ?? 0);

        if ($toggle_id === $current_user_id) {
            flash('error', 'You cannot deactivate your own account.');
        } elseif ($toggle_id > 0) {
            db_execute(
                "UPDATE users SET is_active = NOT is_active WHERE id = ?",
                ['i', $toggle_id]
            );
            flash('success', 'User status updated.');
        }
        csrf_regenerate();
        redirect('users.php');
    }

    /* ── CREATE / UPDATE ───────────────────────────────────── */
    if ($action === 'save') {
        $edit_id   = (int) ($_POST['edit_id'] ?? 0);
        $full_name = trim($_POST['full_name'] ?? '');
        $email     = trim($_POST['email'] ?? '');
        $phone     = trim($_POST['phone'] ?? '');
        $role      = trim($_POST['role'] ?? ROLE_CLIENT);
        $client_id = (int) ($_POST['client_id'] ?? 0);
        $password  = $_POST['password'] ?? '';
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        $errors = [];

        /* Validation */
        if ($full_name === '') {
            $errors[] = 'Full name is required.';
        }
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'A valid email address is required.';
        } else {
            /* Unique check */
            $dup_sql = "SELECT id FROM users WHERE email = ?";
            $dup_params = ['s', $email];
            if ($edit_id > 0) {
                $dup_sql .= " AND id != ?";
                $dup_params[0] .= 'i';
                $dup_params[] = $edit_id;
            }
            if (db_select_one($dup_sql, $dup_params)) {
                $errors[] = 'This email is already in use.';
            }
        }

        if (!in_array($role, [ROLE_ACCOUNTANT, ROLE_CLIENT])) {
            $errors[] = 'Invalid role selected.';
        }

        /* Password required for new users */
        if ($edit_id === 0) {
            if ($password === '') {
                $errors[] = 'Password is required for new users.';
            } elseif (strlen($password) < PASSWORD_MIN_LEN) {
                $errors[] = 'Password must be at least ' . PASSWORD_MIN_LEN . ' characters.';
            }
        } elseif ($password !== '' && strlen($password) < PASSWORD_MIN_LEN) {
            $errors[] = 'Password must be at least ' . PASSWORD_MIN_LEN . ' characters.';
        }

        /* Client link validation */
        $client_val = null;
        if ($role === ROLE_CLIENT && $client_id > 0) {
            $client_val = $client_id;
        }

        /* Prevent self-demotion or self-deactivation */
        if ($edit_id === $current_user_id) {
            if ($role !== ROLE_ACCOUNTANT) {
                $errors[] = 'You cannot change your own role from accountant.';
            }
            if (!$is_active) {
                $errors[] = 'You cannot deactivate your own account.';
            }
        }

        if (!empty($errors)) {
            flash('error', implode(' ', $errors));
        } else {
            if ($edit_id > 0) {
                /* Update existing */
                if ($password !== '') {
                    $hash = password_hash($password, PASSWORD_BCRYPT);
                    db_execute(
                        "UPDATE users SET full_name = ?, email = ?, phone = ?,
                            role = ?, client_id = ?, password_hash = ?, is_active = ?
                         WHERE id = ?",
                        ['ssssissi', $full_name, $email, $phone, $role, $client_val, $hash, $is_active, $edit_id]
                    );
                } else {
                    db_execute(
                        "UPDATE users SET full_name = ?, email = ?, phone = ?,
                            role = ?, client_id = ?, is_active = ?
                         WHERE id = ?",
                        ['ssssiii', $full_name, $email, $phone, $role, $client_val, $is_active, $edit_id]
                    );
                }
                flash('success', 'User account updated.');
            } else {
                /* Create new */
                $hash = password_hash($password, PASSWORD_BCRYPT);
                db_insert(
                    "INSERT INTO users (full_name, email, phone, password_hash, role, client_id, is_active)
                     VALUES (?, ?, ?, ?, ?, ?, ?)",
                    ['sssssii', $full_name, $email, $phone, $hash, $role, $client_val, $is_active]
                );
                flash('success', 'User account created.');
            }
        }

        csrf_regenerate();
        redirect('users.php');
    }
}

/* ══════════════════════════════════════════════════════════════
 *  LOAD DATA
 * ══════════════════════════════════════════════════════════════ */

/* Search & Filter */
$search = trim($_GET['search'] ?? '');
$role_filter = $_GET['role'] ?? 'all';
$status_filter = $_GET['status'] ?? 'all';

$where = [];
$params = [];
$types = '';

if ($search !== '') {
    $safe = db_escape_like($search);
    $where[] = "(u.full_name LIKE CONCAT('%',?,'%') OR u.email LIKE CONCAT('%',?,'%'))";
    $params[] = $safe;
    $params[] = $safe;
    $types .= 'ss';
}
if ($role_filter !== 'all' && in_array($role_filter, [ROLE_ACCOUNTANT, ROLE_CLIENT])) {
    $where[] = "u.role = ?";
    $params[] = $role_filter;
    $types .= 's';
}
if ($status_filter === 'active') {
    $where[] = "u.is_active = 1";
} elseif ($status_filter === 'inactive') {
    $where[] = "u.is_active = 0";
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* Stats */
$sp = empty($types) ? [] : array_merge([$types], $params);
$total_users = (int) db_scalar("SELECT COUNT(*) FROM users u {$where_sql}", $sp);
$total_accountants = (int) db_scalar("SELECT COUNT(*) FROM users WHERE role = ?", ['s', ROLE_ACCOUNTANT]);
$total_clients = (int) db_scalar("SELECT COUNT(*) FROM users WHERE role = ?", ['s', ROLE_CLIENT]);
$active_users = (int) db_scalar("SELECT COUNT(*) FROM users WHERE is_active = 1");

/* Pagination */
$pager = db_paginate("SELECT COUNT(*) FROM users u {$where_sql}", $sp);

$sel_types = $types . 'ii';
$sel_params = array_merge($params, [$pager['limit'], $pager['offset']]);

$users = db_select(
    "SELECT u.*, c.company_name
     FROM users u
     LEFT JOIN clients c ON c.id = u.client_id
     {$where_sql}
     ORDER BY u.created_at DESC
     LIMIT ? OFFSET ?",
    array_merge([$sel_types], $sel_params)
);

/* Client list for linking */
$client_list = db_select(
    "SELECT id, company_name FROM clients WHERE is_active = 1 ORDER BY company_name ASC"
);

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Manage Users';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1><i class="bi bi-people me-2" style="font-size:1.4rem"></i>Manage Users</h1>
        <div class="lp-subtitle">Create and manage user accounts for accountants and client portal access</div>
    </div>
    <button class="btn btn-lp-primary" data-bs-toggle="modal" data-bs-target="#userModal"
            onclick="resetUserForm()">
        <i class="bi bi-plus-lg me-1"></i> Add User
    </button>
</div>

<!-- ═══ Stats ═══════════════════════════════════════════════════ -->
<div class="row g-3 mb-4">
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon blue"><i class="bi bi-people-fill"></i></div>
            <div><div class="lp-stat-label">Total Users</div>
                 <div class="lp-stat-value"><?php echo $total_users; ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon purple"><i class="bi bi-shield-check"></i></div>
            <div><div class="lp-stat-label">Accountants</div>
                 <div class="lp-stat-value"><?php echo $total_accountants; ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon amber"><i class="bi bi-person-badge"></i></div>
            <div><div class="lp-stat-label">Client Users</div>
                 <div class="lp-stat-value"><?php echo $total_clients; ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon emerald"><i class="bi bi-check-circle"></i></div>
            <div><div class="lp-stat-label">Active</div>
                 <div class="lp-stat-value"><?php echo $active_users; ?></div></div>
        </div></div>
    </div>
</div>

<!-- ═══ Filters ═════════════════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="users.php" class="row g-2 align-items-end">
            <div class="col-lg-4 col-md-5">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Search name or email…"
                           value="<?php echo h($search); ?>">
                </div>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <select name="role" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all" <?php echo $role_filter === 'all' ? 'selected' : ''; ?>>All Roles</option>
                    <option value="<?php echo ROLE_ACCOUNTANT; ?>" <?php echo $role_filter === ROLE_ACCOUNTANT ? 'selected' : ''; ?>>Accountants</option>
                    <option value="<?php echo ROLE_CLIENT; ?>" <?php echo $role_filter === ROLE_CLIENT ? 'selected' : ''; ?>>Clients</option>
                </select>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <select name="status" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                    <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                    <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                </select>
            </div>
            <div class="col-lg-4 col-md-4 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary"><i class="bi bi-funnel me-1"></i> Filter</button>
                <?php $hf = ($search !== '' || $role_filter !== 'all' || $status_filter !== 'all');
                if ($hf): ?>
                <a href="users.php" class="btn btn-lp-outline"><i class="bi bi-x-lg me-1"></i> Clear</a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ User List ═══════════════════════════════════════════════ -->
<?php if (empty($users)): ?>
    <div class="lp-card"><div class="lp-empty">
        <i class="bi bi-people d-block"></i>
        <p><?php echo $hf ? 'No users match your filters.' : 'No users found.'; ?></p>
    </div></div>
<?php else: ?>

<!-- Desktop -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>User</th>
                    <th>Email</th>
                    <th>Role</th>
                    <th>Linked Client</th>
                    <th class="text-center">Status</th>
                    <th>Last Login</th>
                    <th class="text-end">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($users as $u):
                    $is_self = ((int)$u['id'] === $current_user_id);
                    $initials = '';
                    foreach (explode(' ', $u['full_name']) as $p) $initials .= strtoupper(substr($p,0,1));
                    $initials = substr($initials, 0, 2);
                ?>
                <tr>
                    <td>
                        <div class="d-flex align-items-center gap-2">
                            <div style="width:36px; height:36px; border-radius:50%;
                                 background:linear-gradient(135deg, var(--lp-navy), var(--lp-blue));
                                 display:flex; align-items:center; justify-content:center;
                                 color:#fff; font-size:.75rem; font-weight:600">
                                <?php echo h($initials); ?>
                            </div>
                            <div>
                                <div class="fw-semibold" style="color:var(--lp-navy); font-size:.9rem">
                                    <?php echo h($u['full_name']); ?>
                                    <?php if ($is_self): ?>
                                        <small class="text-muted">(you)</small>
                                    <?php endif; ?>
                                </div>
                                <?php if ($u['phone']): ?>
                                <small class="text-muted" style="font-size:.72rem"><?php echo h($u['phone']); ?></small>
                                <?php endif; ?>
                            </div>
                        </div>
                    </td>
                    <td style="font-size:.84rem"><?php echo h($u['email']); ?></td>
                    <td>
                        <span class="badge rounded-pill"
                              style="background:<?php echo $u['role'] === ROLE_ACCOUNTANT ? 'var(--lp-purple-soft, #f3e8ff)' : 'var(--lp-amber-soft)'; ?>;
                                     color:<?php echo $u['role'] === ROLE_ACCOUNTANT ? 'var(--lp-purple)' : 'var(--lp-amber)'; ?>;
                                     font-size:.72rem; font-weight:600">
                            <?php echo ucfirst(h($u['role'])); ?>
                        </span>
                    </td>
                    <td style="font-size:.84rem">
                        <?php if ($u['company_name']): ?>
                            <a href="client_view.php?id=<?php echo (int)$u['client_id']; ?>"
                               class="text-decoration-none" style="color:var(--lp-blue)">
                                <?php echo h($u['company_name']); ?>
                            </a>
                        <?php else: ?>
                            <span class="text-muted">—</span>
                        <?php endif; ?>
                    </td>
                    <td class="text-center">
                        <?php if ((int)$u['is_active']): ?>
                            <span class="badge rounded-pill" style="background:var(--lp-emerald-soft); color:var(--lp-emerald); font-size:.68rem">Active</span>
                        <?php else: ?>
                            <span class="badge rounded-pill" style="background:var(--lp-red-soft, #fef2f2); color:var(--lp-red); font-size:.68rem">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td style="font-size:.82rem; color:var(--lp-slate-mid)">
                        <?php echo $u['last_login_at'] ? date(DATETIME_FORMAT, strtotime($u['last_login_at'])) : 'Never'; ?>
                    </td>
                    <td class="text-end">
                        <div class="dropdown">
                            <button class="btn btn-sm btn-lp-outline dropdown-toggle" type="button"
                                    data-bs-toggle="dropdown" style="padding:.3rem .65rem; font-size:.78rem">
                                Actions
                            </button>
                            <ul class="dropdown-menu dropdown-menu-end shadow-sm">
                                <li>
                                    <button class="dropdown-item" onclick="editUser(<?php echo h(json_encode($u)); ?>)">
                                        <i class="bi bi-pencil me-2"></i> Edit
                                    </button>
                                </li>
                                <?php if (!$is_self): ?>
                                <li>
                                    <form method="POST" action="users.php">
                                        <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                                        <input type="hidden" name="action" value="toggle_active">
                                        <input type="hidden" name="user_id" value="<?php echo (int)$u['id']; ?>">
                                        <button type="submit" class="dropdown-item">
                                            <i class="bi bi-<?php echo (int)$u['is_active'] ? 'x-circle' : 'check-circle'; ?> me-2"></i>
                                            <?php echo (int)$u['is_active'] ? 'Deactivate' : 'Activate'; ?>
                                        </button>
                                    </form>
                                </li>
                                <li><hr class="dropdown-divider"></li>
                                <li>
                                    <form method="POST" action="users.php"
                                          data-confirm="Delete user <?php echo h($u['full_name']); ?>? This cannot be undone.">
                                        <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="user_id" value="<?php echo (int)$u['id']; ?>">
                                        <button type="submit" class="dropdown-item text-danger">
                                            <i class="bi bi-trash me-2"></i> Delete
                                        </button>
                                    </form>
                                </li>
                                <?php endif; ?>
                            </ul>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile -->
<div class="d-md-none mb-4">
    <?php foreach ($users as $u):
        $is_self = ((int)$u['id'] === $current_user_id);
        $initials = '';
        foreach (explode(' ', $u['full_name']) as $p) $initials .= strtoupper(substr($p,0,1));
        $initials = substr($initials, 0, 2);
    ?>
    <div class="lp-card mb-3"><div class="card-body">
        <div class="d-flex align-items-start gap-3 mb-3">
            <div style="width:44px; height:44px; border-radius:50%; flex-shrink:0;
                 background:linear-gradient(135deg, var(--lp-navy), var(--lp-blue));
                 display:flex; align-items:center; justify-content:center;
                 color:#fff; font-size:.85rem; font-weight:600">
                <?php echo h($initials); ?>
            </div>
            <div class="flex-grow-1">
                <div class="fw-semibold" style="color:var(--lp-navy)">
                    <?php echo h($u['full_name']); ?>
                    <?php if ($is_self): ?><small class="text-muted">(you)</small><?php endif; ?>
                </div>
                <small class="text-muted d-block"><?php echo h($u['email']); ?></small>
                <div class="d-flex gap-2 mt-1">
                    <span class="badge rounded-pill" style="background:<?php echo $u['role'] === ROLE_ACCOUNTANT ? 'var(--lp-purple-soft, #f3e8ff)' : 'var(--lp-amber-soft)'; ?>; color:<?php echo $u['role'] === ROLE_ACCOUNTANT ? 'var(--lp-purple)' : 'var(--lp-amber)'; ?>; font-size:.68rem">
                        <?php echo ucfirst(h($u['role'])); ?>
                    </span>
                    <?php if ((int)$u['is_active']): ?>
                        <span class="badge rounded-pill" style="background:var(--lp-emerald-soft); color:var(--lp-emerald); font-size:.68rem">Active</span>
                    <?php else: ?>
                        <span class="badge rounded-pill" style="background:var(--lp-red-soft, #fef2f2); color:var(--lp-red); font-size:.68rem">Inactive</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <div class="d-flex gap-2 pt-2" style="border-top:1px solid var(--lp-silver)">
            <button class="btn btn-sm btn-lp-outline" style="font-size:.78rem"
                    onclick="editUser(<?php echo h(json_encode($u)); ?>)">
                <i class="bi bi-pencil me-1"></i> Edit
            </button>
            <?php if (!$is_self): ?>
            <form method="POST" action="users.php" class="d-inline">
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="toggle_active">
                <input type="hidden" name="user_id" value="<?php echo (int)$u['id']; ?>">
                <button type="submit" class="btn btn-sm btn-lp-outline" style="font-size:.78rem">
                    <?php echo (int)$u['is_active'] ? 'Deactivate' : 'Activate'; ?>
                </button>
            </form>
            <form method="POST" action="users.php" class="d-inline" data-confirm="Delete this user?">
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="user_id" value="<?php echo (int)$u['id']; ?>">
                <button type="submit" class="btn btn-sm text-danger border-0" style="font-size:.78rem">
                    <i class="bi bi-trash me-1"></i> Delete
                </button>
            </form>
            <?php endif; ?>
        </div>
    </div></div>
    <?php endforeach; ?>
</div>

<!-- Pagination -->
<?php if ($pager['total_pages'] > 1): ?>
<nav><ul class="pagination lp-pagination justify-content-center mb-0">
    <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>"><i class="bi bi-chevron-left"></i></a></li>
    <?php
    $rng = 2; $s = max(1, $pager['current']-$rng); $e = min($pager['total_pages'], $pager['current']+$rng);
    if ($s > 1): ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>1])); ?>">1</a></li><?php if ($s > 2): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; endif;
    for ($p=$s; $p<=$e; $p++): ?><li class="page-item <?php echo $p===$pager['current']?'active':''; ?>"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$p])); ?>"><?php echo $p; ?></a></li><?php endfor;
    if ($e < $pager['total_pages']): if ($e < $pager['total_pages']-1): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$pager['total_pages']])); ?>"><?php echo $pager['total_pages']; ?></a></li><?php endif; ?>
    <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>"><i class="bi bi-chevron-right"></i></a></li>
</ul></nav>
<?php endif; ?>

<?php endif; ?>

<!-- ═══ Create / Edit Modal ═════════════════════════════════════ -->
<div class="modal fade" id="userModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content" style="border:0; border-radius:var(--lp-radius-lg); overflow:hidden">
            <form method="POST" action="users.php" class="lp-form" novalidate>
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="save">
                <input type="hidden" name="edit_id" id="user_edit_id" value="0">

                <div class="modal-header" style="background:var(--lp-snow); border-bottom:1px solid var(--lp-silver)">
                    <h5 class="modal-title" id="userModalLabel">
                        <i class="bi bi-person-plus me-1"></i> Add User
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Full Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="full_name" id="user_full_name"
                                   required maxlength="100">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email Address <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="email" id="user_email"
                                   required maxlength="150">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Phone</label>
                            <input type="tel" class="form-control" name="phone" id="user_phone"
                                   maxlength="30" placeholder="Optional">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Role <span class="text-danger">*</span></label>
                            <select class="form-select" name="role" id="user_role" required>
                                <option value="<?php echo ROLE_ACCOUNTANT; ?>">Accountant</option>
                                <option value="<?php echo ROLE_CLIENT; ?>">Client</option>
                            </select>
                        </div>
                        <div class="col-md-6" id="clientLinkWrap" style="display:none">
                            <label class="form-label">Link to Client</label>
                            <select class="form-select" name="client_id" id="user_client_id">
                                <option value="0">— Select client —</option>
                                <?php foreach ($client_list as $cl): ?>
                                <option value="<?php echo (int)$cl['id']; ?>"><?php echo h($cl['company_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted" style="font-size:.72rem">Link this user to a client record for portal access.</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label" id="passwordLabel">Password <span class="text-danger">*</span></label>
                            <input type="password" class="form-control" name="password" id="user_password"
                                   minlength="<?php echo PASSWORD_MIN_LEN; ?>" autocomplete="new-password">
                            <small class="text-muted" style="font-size:.72rem" id="passwordHint">
                                Minimum <?php echo PASSWORD_MIN_LEN; ?> characters
                            </small>
                        </div>
                        <div class="col-12">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" role="switch"
                                       id="user_is_active" name="is_active" value="1" checked
                                       style="width:2.5em; height:1.25em; cursor:pointer">
                                <label class="form-check-label ms-1" for="user_is_active"
                                       style="font-weight:500; cursor:pointer; font-size:.88rem">
                                    Account is active
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer" style="background:var(--lp-snow); border-top:1px solid var(--lp-silver)">
                    <button type="button" class="btn btn-lp-outline" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i> <span id="userSaveLabel">Create User</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
/* Show/hide client link based on role */
document.getElementById('user_role').addEventListener('change', function() {
    document.getElementById('clientLinkWrap').style.display =
        (this.value === '<?php echo ROLE_CLIENT; ?>') ? '' : 'none';
});

function resetUserForm() {
    document.getElementById('user_edit_id').value    = '0';
    document.getElementById('user_full_name').value  = '';
    document.getElementById('user_email').value      = '';
    document.getElementById('user_phone').value      = '';
    document.getElementById('user_role').value       = '<?php echo ROLE_CLIENT; ?>';
    document.getElementById('user_client_id').value  = '0';
    document.getElementById('user_password').value   = '';
    document.getElementById('user_password').required = true;
    document.getElementById('user_is_active').checked = true;
    document.getElementById('clientLinkWrap').style.display = '';
    document.getElementById('userModalLabel').innerHTML = '<i class="bi bi-person-plus me-1"></i> Add User';
    document.getElementById('userSaveLabel').textContent = 'Create User';
    document.getElementById('passwordLabel').innerHTML = 'Password <span class="text-danger">*</span>';
    document.getElementById('passwordHint').textContent = 'Minimum <?php echo PASSWORD_MIN_LEN; ?> characters';
}

function editUser(data) {
    document.getElementById('user_edit_id').value    = data.id;
    document.getElementById('user_full_name').value  = data.full_name;
    document.getElementById('user_email').value      = data.email;
    document.getElementById('user_phone').value      = data.phone || '';
    document.getElementById('user_role').value       = data.role;
    document.getElementById('user_client_id').value  = data.client_id || '0';
    document.getElementById('user_password').value   = '';
    document.getElementById('user_password').required = false;
    document.getElementById('user_is_active').checked = parseInt(data.is_active) === 1;
    document.getElementById('clientLinkWrap').style.display =
        (data.role === '<?php echo ROLE_CLIENT; ?>') ? '' : 'none';
    document.getElementById('userModalLabel').innerHTML = '<i class="bi bi-pencil me-1"></i> Edit User';
    document.getElementById('userSaveLabel').textContent = 'Update User';
    document.getElementById('passwordLabel').innerHTML = 'New Password <small class="text-muted">(leave blank to keep current)</small>';
    document.getElementById('passwordHint').textContent = 'Leave blank to keep existing password';
    new bootstrap.Modal(document.getElementById('userModal')).show();
}
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
