<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  payments.php — Payment Management List
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - Search by invoice number or client name
 *    - Filter by payment method
 *    - Date range filter
 *    - Summary stats: total collected, this month, payment count
 *    - Delete with CSRF + invoice balance recalculation
 *    - Desktop table + mobile card layout
 *
 *  Authorization:
 *    - Accountant: all payments
 *    - Client: own payments only
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_login();

/* ══════════════════════════════════════════════════════════════
 *  HANDLE DELETE (POST, accountant only)
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete' && is_accountant()) {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('payments.php');
    }

    $del_id = (int) ($_POST['payment_id'] ?? 0);

    if ($del_id > 0) {
        /* Load payment to recalculate invoice after deletion */
        $pay = db_select_one("SELECT invoice_id, amount FROM payments WHERE id = ?", ['i', $del_id]);

        if ($pay) {
            db_begin();
            try {
                db_execute("DELETE FROM payments WHERE id = ?", ['i', $del_id]);

                /* Recalculate invoice amount_paid & balance_due */
                $inv_id     = (int) $pay['invoice_id'];
                $new_paid   = (float) db_scalar(
                    "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE invoice_id = ?",
                    ['i', $inv_id]
                );
                $inv_total  = (float) db_scalar(
                    "SELECT total FROM invoices WHERE id = ?",
                    ['i', $inv_id]
                );
                $new_balance = round($inv_total - $new_paid, 2);

                db_execute(
                    "UPDATE invoices SET amount_paid = ?, balance_due = ? WHERE id = ?",
                    ['ddi', $new_paid, $new_balance, $inv_id]
                );

                db_commit();
                flash('success', 'Payment deleted and invoice balance recalculated.');
            } catch (Exception $e) {
                db_rollback();
                flash('error', 'Error deleting payment: ' . $e->getMessage());
            }
        }
    }

    csrf_regenerate();
    redirect('payments.php');
}

/* ══════════════════════════════════════════════════════════════
 *  FILTERS & SEARCH
 * ══════════════════════════════════════════════════════════════ */
$search    = trim($_GET['search'] ?? '');
$method    = $_GET['method']    ?? 'all';
$date_from = trim($_GET['date_from'] ?? '');
$date_to   = trim($_GET['date_to']   ?? '');

$where  = [];
$params = [];
$types  = '';

/* Client role lock */
if (is_client()) {
    $cid      = (int) ($_SESSION['client_id'] ?? 0);
    $where[]  = "p.client_id = ?";
    $params[] = $cid;
    $types   .= 'i';
}

/* Search */
if ($search !== '') {
    $safe     = db_escape_like($search);
    $where[]  = "(i.invoice_number LIKE CONCAT('%',?,'%') OR c.company_name LIKE CONCAT('%',?,'%'))";
    $params[] = $safe;
    $params[] = $safe;
    $types   .= 'ss';
}

/* Method filter */
$valid_methods = ['cash','bank_transfer','check','credit_card','paypal','other'];
if ($method !== 'all' && in_array($method, $valid_methods)) {
    $where[]  = "p.method = ?";
    $params[] = $method;
    $types   .= 's';
}

/* Date range */
if ($date_from !== '') {
    $where[]  = "p.payment_date >= ?";
    $params[] = $date_from;
    $types   .= 's';
}
if ($date_to !== '') {
    $where[]  = "p.payment_date <= ?";
    $params[] = $date_to;
    $types   .= 's';
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* ── Summary Stats ────────────────────────────────────────────── */
$stat_params = empty($types) ? [] : array_merge([$types], $params);

$stat_total = (float) db_scalar(
    "SELECT COALESCE(SUM(p.amount), 0)
     FROM payments p JOIN invoices i ON i.id = p.invoice_id JOIN clients c ON c.id = p.client_id
     {$where_sql}",
    $stat_params
);

$stat_count = (int) db_scalar(
    "SELECT COUNT(*)
     FROM payments p JOIN invoices i ON i.id = p.invoice_id JOIN clients c ON c.id = p.client_id
     {$where_sql}",
    $stat_params
);

/* This month (unfiltered for role, but role-locked) */
$month_where = is_client()
    ? "WHERE p.client_id = ? AND p.payment_date >= ?"
    : "WHERE p.payment_date >= ?";
$month_params = is_client()
    ? ['is', (int)($_SESSION['client_id'] ?? 0), date('Y-m-01')]
    : ['s', date('Y-m-01')];

$stat_this_month = (float) db_scalar(
    "SELECT COALESCE(SUM(p.amount), 0) FROM payments p {$month_where}",
    $month_params
);

/* ── Pagination ───────────────────────────────────────────────── */
$count_params = empty($types) ? [] : array_merge([$types], $params);
$pager = db_paginate(
    "SELECT COUNT(*)
     FROM payments p JOIN invoices i ON i.id = p.invoice_id JOIN clients c ON c.id = p.client_id
     {$where_sql}",
    $count_params
);

/* ── Fetch Payments ───────────────────────────────────────────── */
$sel_types  = $types . 'ii';
$sel_params = array_merge($params, [$pager['limit'], $pager['offset']]);

$payments = db_select(
    "SELECT p.*, i.invoice_number, c.company_name
     FROM payments p
     JOIN invoices i ON i.id = p.invoice_id
     JOIN clients c  ON c.id = p.client_id
     {$where_sql}
     ORDER BY p.payment_date DESC, p.id DESC
     LIMIT ? OFFSET ?",
    array_merge([$sel_types], $sel_params)
);

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = is_accountant() ? 'Payments' : 'My Payments';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <i class="bi bi-credit-card me-2" style="font-size:1.4rem"></i>
            <?php echo is_accountant() ? 'Payments' : 'My Payments'; ?>
        </h1>
        <div class="lp-subtitle">
            <?php echo $pager['total_rows']; ?> payment<?php echo $pager['total_rows'] !== 1 ? 's' : ''; ?> recorded
        </div>
    </div>
    <?php if (is_accountant()): ?>
    <a href="<?php echo APP_URL; ?>/payment_form.php" class="btn btn-lp-primary">
        <i class="bi bi-plus-lg me-1"></i> Record Payment
    </a>
    <?php endif; ?>
</div>

<!-- ═══ Summary Stats ═══════════════════════════════════════════ -->
<div class="row g-3 mb-4">
    <div class="col-sm-4">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon emerald"><i class="bi bi-cash-stack"></i></div>
                <div>
                    <div class="lp-stat-label">Total Collected</div>
                    <div class="lp-stat-value"><?php echo format_currency($stat_total); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-sm-4">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon blue"><i class="bi bi-calendar-check"></i></div>
                <div>
                    <div class="lp-stat-label">This Month</div>
                    <div class="lp-stat-value"><?php echo format_currency($stat_this_month); ?></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-sm-4">
        <div class="lp-card h-100">
            <div class="lp-stat">
                <div class="lp-stat-icon purple"><i class="bi bi-hash"></i></div>
                <div>
                    <div class="lp-stat-label">Transactions</div>
                    <div class="lp-stat-value"><?php echo $stat_count; ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- ═══ Search & Filters ════════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="payments.php" class="row g-2 align-items-end">
            <!-- Search -->
            <div class="col-lg-3 col-md-6">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Invoice # or client…"
                           value="<?php echo h($search); ?>">
                </div>
            </div>

            <!-- Method -->
            <div class="col-lg-2 col-md-3 col-6">
                <select name="method" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all">All Methods</option>
                    <?php
                    $method_labels = [
                        'cash'          => 'Cash',
                        'bank_transfer' => 'Bank Transfer',
                        'check'         => 'Check',
                        'credit_card'   => 'Credit Card',
                        'paypal'        => 'PayPal',
                        'other'         => 'Other',
                    ];
                    foreach ($method_labels as $val => $label):
                    ?>
                    <option value="<?php echo $val; ?>" <?php echo $method === $val ? 'selected' : ''; ?>>
                        <?php echo $label; ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- Date range -->
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_from" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_from); ?>" title="From date">
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_to" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_to); ?>" title="To date">
            </div>

            <!-- Buttons -->
            <div class="col-lg-3 col-md-3 col-6 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary">
                    <i class="bi bi-funnel me-1"></i> Filter
                </button>
                <?php
                $has_filters = ($search !== '' || $method !== 'all' || $date_from !== '' || $date_to !== '');
                if ($has_filters):
                ?>
                <a href="<?php echo APP_URL; ?>/payments.php" class="btn btn-lp-outline">
                    <i class="bi bi-x-lg me-1"></i> Clear
                </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ Payment List ════════════════════════════════════════════ -->
<?php if (empty($payments)): ?>
    <div class="lp-card">
        <div class="lp-empty">
            <i class="bi bi-credit-card d-block"></i>
            <p>
                <?php echo $has_filters
                    ? 'No payments match your filters.'
                    : 'No payments recorded yet.'; ?>
            </p>
            <?php if (!$has_filters && is_accountant()): ?>
            <a href="<?php echo APP_URL; ?>/payment_form.php" class="btn btn-lp-primary mt-3">
                <i class="bi bi-plus-lg me-1"></i> Record Payment
            </a>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>

<!-- Desktop table -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Invoice</th>
                    <?php if (is_accountant()): ?><th>Client</th><?php endif; ?>
                    <th>Method</th>
                    <th>Reference</th>
                    <th class="text-end">Amount</th>
                    <?php if (is_accountant()): ?><th class="text-end">Actions</th><?php endif; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($payments as $pay): ?>
                <tr>
                    <td style="font-size:.84rem">
                        <?php echo date(DATE_FORMAT, strtotime($pay['payment_date'])); ?>
                    </td>
                    <td>
                        <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$pay['invoice_id']; ?>"
                           class="fw-semibold text-decoration-none" style="color:var(--lp-navy)">
                            <?php echo h($pay['invoice_number']); ?>
                        </a>
                    </td>
                    <?php if (is_accountant()): ?>
                    <td>
                        <a href="<?php echo APP_URL; ?>/client_view.php?id=<?php echo (int)$pay['client_id']; ?>"
                           class="text-decoration-none" style="color:var(--lp-blue); font-size:.84rem">
                            <?php echo h($pay['company_name']); ?>
                        </a>
                    </td>
                    <?php endif; ?>
                    <td>
                        <span class="badge rounded-pill"
                              style="background:var(--lp-snow); color:var(--lp-slate);
                                     border:1px solid var(--lp-silver); font-size:.72rem; font-weight:500">
                            <?php echo h(ucfirst(str_replace('_', ' ', $pay['method']))); ?>
                        </span>
                    </td>
                    <td style="font-size:.84rem; color:var(--lp-slate-mid)">
                        <?php echo h($pay['reference'] ?: '—'); ?>
                    </td>
                    <td class="text-end fw-semibold" style="color:var(--lp-emerald)">
                        +<?php echo format_currency((float)$pay['amount']); ?>
                    </td>
                    <?php if (is_accountant()): ?>
                    <td class="text-end">
                        <form method="POST" action="payments.php" class="d-inline"
                              data-confirm="Delete this payment of <?php echo format_currency((float)$pay['amount']); ?>? The invoice balance will be recalculated.">
                            <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                                   value="<?php echo csrf_token(); ?>">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="payment_id" value="<?php echo (int)$pay['id']; ?>">
                            <button type="submit" class="btn btn-sm text-danger border-0"
                                    style="padding:.2rem .5rem; font-size:.8rem" title="Delete payment">
                                <i class="bi bi-trash"></i>
                            </button>
                        </form>
                    </td>
                    <?php endif; ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile cards -->
<div class="d-md-none mb-4">
    <?php foreach ($payments as $pay): ?>
    <div class="lp-card mb-3">
        <div class="card-body">
            <div class="d-flex align-items-start justify-content-between mb-2">
                <div>
                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo (int)$pay['invoice_id']; ?>"
                       class="fw-semibold text-decoration-none d-block" style="color:var(--lp-navy); font-size:.95rem">
                        <?php echo h($pay['invoice_number']); ?>
                    </a>
                    <small class="text-muted"><?php echo h($pay['company_name']); ?></small>
                </div>
                <span class="fw-bold" style="color:var(--lp-emerald); font-size:1rem">
                    +<?php echo format_currency((float)$pay['amount']); ?>
                </span>
            </div>
            <div class="d-flex gap-3 mb-2" style="font-size:.82rem">
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Date</small>
                    <?php echo date(DATE_FORMAT, strtotime($pay['payment_date'])); ?>
                </div>
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Method</small>
                    <?php echo h(ucfirst(str_replace('_', ' ', $pay['method']))); ?>
                </div>
                <?php if ($pay['reference']): ?>
                <div>
                    <small class="text-muted d-block" style="font-size:.68rem; text-transform:uppercase; letter-spacing:.04em">Ref</small>
                    <?php echo h($pay['reference']); ?>
                </div>
                <?php endif; ?>
            </div>
            <?php if (is_accountant()): ?>
            <div class="pt-2" style="border-top:1px solid var(--lp-silver)">
                <form method="POST" action="payments.php" class="d-inline"
                      data-confirm="Delete this payment?">
                    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                           value="<?php echo csrf_token(); ?>">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="payment_id" value="<?php echo (int)$pay['id']; ?>">
                    <button type="submit" class="btn btn-sm text-danger border-0" style="font-size:.82rem">
                        <i class="bi bi-trash me-1"></i> Delete
                    </button>
                </form>
            </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- ═══ Pagination ══════════════════════════════════════════════ -->
<?php if ($pager['total_pages'] > 1): ?>
<nav aria-label="Payment pagination">
    <ul class="pagination lp-pagination justify-content-center mb-0">
        <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>">
                <i class="bi bi-chevron-left"></i>
            </a>
        </li>
        <?php
        $range = 2;
        $start = max(1, $pager['current'] - $range);
        $end   = min($pager['total_pages'], $pager['current'] + $range);
        if ($start > 1): ?>
            <li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => 1])); ?>">1</a></li>
            <?php if ($start > 2): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif;
        endif;
        for ($p = $start; $p <= $end; $p++): ?>
            <li class="page-item <?php echo $p === $pager['current'] ? 'active' : ''; ?>">
                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $p])); ?>"><?php echo $p; ?></a>
            </li>
        <?php endfor;
        if ($end < $pager['total_pages']): ?>
            <?php if ($end < $pager['total_pages'] - 1): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; ?>
            <li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['total_pages']])); ?>"><?php echo $pager['total_pages']; ?></a></li>
        <?php endif; ?>
        <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>">
                <i class="bi bi-chevron-right"></i>
            </a>
        </li>
    </ul>
</nav>
<?php endif; ?>

<?php endif; ?>

<?php require_once __DIR__ . '/footer.php'; ?>
