<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  payment_form.php — Record a Payment
 * ═══════════════════════════════════════════════════════════════
 *
 *  Entry points:
 *    GET  /payment_form.php                     → blank form
 *    GET  /payment_form.php?invoice_id=5        → pre-select invoice
 *    GET  /payment_form.php?client_id=3         → filter invoices to client
 *    POST /payment_form.php                     → validate + save
 *
 *  Save logic (transactional):
 *    1. Insert payment record
 *    2. Recalculate invoice amount_paid & balance_due
 *    3. Auto-set invoice status to 'paid' if balance reaches zero
 *
 *  Security:
 *    - Accountant only
 *    - CSRF validated
 *    - Overpayment warning (soft — allows with confirmation)
 *    - Prepared statements throughout
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

$errors = [];

/* ── Default field values ─────────────────────────────────────── */
$f = [
    'invoice_id'   => (int) ($_GET['invoice_id'] ?? 0),
    'amount'       => '',
    'payment_date' => date('Y-m-d'),
    'method'       => 'bank_transfer',
    'reference'    => '',
    'notes'        => '',
];

/* ── Pre-fill amount from invoice balance if invoice_id given ─── */
$prefill_invoice = null;
if ($f['invoice_id'] > 0) {
    $prefill_invoice = db_select_one(
        "SELECT i.id, i.invoice_number, i.total, i.balance_due, i.status,
                c.company_name, c.id AS client_id
         FROM invoices i
         JOIN clients c ON c.id = i.client_id
         WHERE i.id = ?",
        ['i', $f['invoice_id']]
    );
    if ($prefill_invoice) {
        $f['amount'] = number_format((float)$prefill_invoice['balance_due'], 2, '.', '');
    }
}

/* ── Build invoice list for dropdown ──────────────────────────── */
$client_filter = (int) ($_GET['client_id'] ?? 0);

$inv_where = "WHERE i.status NOT IN ('paid','cancelled')";
$inv_params = [];
$inv_types  = '';

if ($client_filter > 0) {
    $inv_where .= " AND i.client_id = ?";
    $inv_params[] = $client_filter;
    $inv_types .= 'i';
}

$inv_list_params = empty($inv_types) ? [] : array_merge([$inv_types], $inv_params);
$invoice_list = db_select(
    "SELECT i.id, i.invoice_number, i.total, i.balance_due, c.company_name
     FROM invoices i
     JOIN clients c ON c.id = i.client_id
     {$inv_where}
     ORDER BY i.created_at DESC",
    $inv_list_params
);

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Validate & Save
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    /* ── CSRF ──────────────────────────────────────────────── */
    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        $errors[] = 'Invalid security token.';
    }

    /* ── Collect ───────────────────────────────────────────── */
    $f['invoice_id']   = (int) ($_POST['invoice_id'] ?? 0);
    $f['amount']       = trim($_POST['amount'] ?? '');
    $f['payment_date'] = trim($_POST['payment_date'] ?? '');
    $f['method']       = trim($_POST['method'] ?? 'bank_transfer');
    $f['reference']    = trim($_POST['reference'] ?? '');
    $f['notes']        = trim($_POST['notes'] ?? '');

    /* ── Validate ──────────────────────────────────────────── */
    if ($f['invoice_id'] <= 0) {
        $errors[] = 'Please select an invoice.';
    } else {
        /* Verify invoice exists and is payable */
        $target_inv = db_select_one(
            "SELECT i.id, i.total, i.balance_due, i.status, i.client_id
             FROM invoices i WHERE i.id = ?",
            ['i', $f['invoice_id']]
        );
        if (!$target_inv) {
            $errors[] = 'Selected invoice not found.';
        } elseif ($target_inv['status'] === 'cancelled') {
            $errors[] = 'Cannot record payment against a cancelled invoice.';
        }
    }

    if ($f['amount'] === '' || !is_numeric($f['amount'])) {
        $errors[] = 'A valid payment amount is required.';
    } elseif ((float) $f['amount'] <= 0) {
        $errors[] = 'Payment amount must be greater than zero.';
    }

    if ($f['payment_date'] === '' || !strtotime($f['payment_date'])) {
        $errors[] = 'A valid payment date is required.';
    }

    $valid_methods = ['cash','bank_transfer','check','credit_card','paypal','other'];
    if (!in_array($f['method'], $valid_methods)) {
        $errors[] = 'Invalid payment method.';
    }

    /* ── Overpayment check (warning, not blocking) ─────────── */
    $overpay_warning = false;
    if (empty($errors) && isset($target_inv)) {
        $amt = (float) $f['amount'];
        $bal = (float) $target_inv['balance_due'];
        if ($amt > $bal && $bal > 0) {
            $overpay_warning = true;
            /* Only block if user hasn't confirmed */
            if (!isset($_POST['confirm_overpay'])) {
                $errors[] = 'Payment of ' . format_currency($amt)
                    . ' exceeds the balance due of ' . format_currency($bal)
                    . '. Check the "Confirm overpayment" box to proceed.';
            }
        }
    }

    /* ── Save ──────────────────────────────────────────────── */
    if (empty($errors)) {
        db_begin();
        try {
            $amount     = round((float)$f['amount'], 2);
            $client_id  = (int) $target_inv['client_id'];
            $invoice_id = (int) $f['invoice_id'];

            /* 1. Insert payment */
            $payment_id = db_insert(
                "INSERT INTO payments
                    (invoice_id, client_id, amount, payment_date, method, reference, notes, recorded_by)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                [
                    'iidsssi',  // Note: 'i' for invoice_id, 'i' for client_id, 'd' for amount
                    $invoice_id, $client_id, $amount,
                    $f['payment_date'], $f['method'], $f['reference'], $f['notes'],
                    $_SESSION['user_id']
                ]
            );

            /* 2. Recalculate invoice */
            $new_paid = (float) db_scalar(
                "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE invoice_id = ?",
                ['i', $invoice_id]
            );
            $inv_total   = (float) $target_inv['total'];
            $new_balance = round($inv_total - $new_paid, 2);
            if ($new_balance < 0) $new_balance = 0;

            /* 3. Auto-update status to 'paid' if fully paid */
            $new_status = $target_inv['status'];
            if ($new_balance <= 0 && !in_array($new_status, ['paid','cancelled'])) {
                $new_status = 'paid';
            }

            db_execute(
                "UPDATE invoices SET amount_paid = ?, balance_due = ?, status = ? WHERE id = ?",
                ['ddsi', $new_paid, $new_balance, $new_status, $invoice_id]
            );

            db_commit();
            csrf_regenerate();

            $msg = 'Payment of ' . format_currency($amount) . ' recorded successfully.';
            if ($new_status === 'paid' && $target_inv['status'] !== 'paid') {
                $msg .= ' Invoice has been marked as paid.';
            }
            flash('success', $msg);
            redirect('invoice_view.php?id=' . $invoice_id);

        } catch (Exception $e) {
            db_rollback();
            $errors[] = 'Database error: ' . $e->getMessage();
        }
    }
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Record Payment';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <i class="bi bi-credit-card me-2" style="font-size:1.4rem"></i>
            Record Payment
        </h1>
        <div class="lp-subtitle">Record a payment received against an invoice.</div>
    </div>
    <a href="<?php echo APP_URL; ?>/payments.php" class="btn btn-lp-outline">
        <i class="bi bi-arrow-left me-1"></i> Back to Payments
    </a>
</div>

<!-- ═══ Errors ══════════════════════════════════════════════════ -->
<?php if (!empty($errors)): ?>
<div class="lp-flash error" style="animation:none;">
    <div>
        <i class="bi bi-exclamation-triangle-fill me-1"></i>
        <strong>Please correct the following:</strong>
        <ul class="mb-0 mt-1 ps-3" style="font-weight:400">
            <?php foreach ($errors as $err): ?>
            <li><?php echo h($err); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
</div>
<?php endif; ?>

<!-- ═══ Payment Form ════════════════════════════════════════════ -->
<form method="POST" action="payment_form.php" class="lp-form" novalidate>
    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
           value="<?php echo csrf_token(); ?>">

    <div class="row g-4">

        <!-- ═══ Left: Form Fields ═══════════════════════════════ -->
        <div class="col-lg-8">
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-cash me-1"></i> Payment Details
                </div>
                <div class="card-body">
                    <div class="row g-3">

                        <!-- Invoice selector -->
                        <div class="col-md-8">
                            <label for="invoice_id" class="form-label">
                                Invoice <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" id="invoice_id" name="invoice_id" required>
                                <option value="0">— Select an unpaid invoice —</option>
                                <?php foreach ($invoice_list as $inv): ?>
                                <option value="<?php echo (int)$inv['id']; ?>"
                                        data-balance="<?php echo (float)$inv['balance_due']; ?>"
                                    <?php echo $f['invoice_id'] === (int)$inv['id'] ? 'selected' : ''; ?>>
                                    <?php echo h($inv['invoice_number']); ?>
                                    — <?php echo h($inv['company_name']); ?>
                                    (Balance: <?php echo format_currency((float)$inv['balance_due']); ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Amount -->
                        <div class="col-md-4">
                            <label for="amount" class="form-label">
                                Amount <span class="text-danger">*</span>
                            </label>
                            <div class="input-group">
                                <span class="input-group-text"><?php echo CURRENCY_SYMBOL; ?></span>
                                <input type="number" class="form-control" id="amount" name="amount"
                                       value="<?php echo h($f['amount']); ?>"
                                       step="0.01" min="0.01" required
                                       data-currency="true"
                                       placeholder="0.00">
                            </div>
                        </div>

                        <!-- Payment date -->
                        <div class="col-md-4">
                            <label for="payment_date" class="form-label">
                                Payment Date <span class="text-danger">*</span>
                            </label>
                            <input type="date" class="form-control" id="payment_date" name="payment_date"
                                   value="<?php echo h($f['payment_date']); ?>" required>
                        </div>

                        <!-- Method -->
                        <div class="col-md-4">
                            <label for="method" class="form-label">
                                Payment Method <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" id="method" name="method" required>
                                <?php
                                $methods = [
                                    'bank_transfer' => 'Bank Transfer',
                                    'cash'          => 'Cash',
                                    'check'         => 'Check',
                                    'credit_card'   => 'Credit Card',
                                    'paypal'        => 'PayPal',
                                    'other'         => 'Other',
                                ];
                                foreach ($methods as $val => $label):
                                ?>
                                <option value="<?php echo $val; ?>"
                                    <?php echo $f['method'] === $val ? 'selected' : ''; ?>>
                                    <?php echo $label; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Reference -->
                        <div class="col-md-4">
                            <label for="reference" class="form-label">Reference / Check #</label>
                            <input type="text" class="form-control" id="reference" name="reference"
                                   value="<?php echo h($f['reference']); ?>"
                                   placeholder="e.g. CHK-4521" maxlength="100">
                        </div>

                        <!-- Notes -->
                        <div class="col-12">
                            <label for="notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="notes" name="notes" rows="3"
                                      style="resize:vertical"
                                      placeholder="Optional notes about this payment…"
                            ><?php echo h($f['notes']); ?></textarea>
                        </div>

                        <!-- Overpayment confirm (hidden unless needed) -->
                        <div class="col-12" id="overpayWrap" style="display:none">
                            <div class="form-check" style="background:var(--lp-amber-soft); padding:.75rem 1rem .75rem 2.5rem; border-radius:var(--lp-radius)">
                                <input class="form-check-input" type="checkbox"
                                       id="confirm_overpay" name="confirm_overpay" value="1">
                                <label class="form-check-label" for="confirm_overpay"
                                       style="font-size:.84rem; font-weight:500; color:var(--lp-amber); cursor:pointer">
                                    <i class="bi bi-exclamation-triangle me-1"></i>
                                    I confirm this payment exceeds the invoice balance.
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- ═══ Right: Sidebar ══════════════════════════════════ -->
        <div class="col-lg-4">

            <!-- Invoice Preview (updates via JS) -->
            <div class="lp-card mb-4" id="invoicePreview">
                <div class="card-header">
                    <i class="bi bi-receipt me-1"></i> Invoice Preview
                </div>
                <div class="card-body" id="previewBody">
                    <?php if ($prefill_invoice): ?>
                    <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                        <span class="text-muted">Invoice</span>
                        <span class="fw-semibold"><?php echo h($prefill_invoice['invoice_number']); ?></span>
                    </div>
                    <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                        <span class="text-muted">Client</span>
                        <span class="fw-semibold"><?php echo h($prefill_invoice['company_name']); ?></span>
                    </div>
                    <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                        <span class="text-muted">Total</span>
                        <span><?php echo format_currency((float)$prefill_invoice['total']); ?></span>
                    </div>
                    <hr class="my-2" style="border-color:var(--lp-silver)">
                    <div class="d-flex justify-content-between" style="font-size:1rem">
                        <span class="fw-bold" style="color:var(--lp-amber)">Balance Due</span>
                        <span class="fw-bold" style="color:var(--lp-amber)">
                            <?php echo format_currency((float)$prefill_invoice['balance_due']); ?>
                        </span>
                    </div>
                    <?php else: ?>
                    <div class="text-center text-muted py-2" style="font-size:.85rem">
                        <i class="bi bi-receipt d-block mb-1" style="font-size:1.5rem; opacity:.4"></i>
                        Select an invoice to see details.
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Actions -->
            <div class="lp-card">
                <div class="card-body d-grid gap-2">
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i> Record Payment
                    </button>
                    <a href="<?php echo APP_URL; ?>/payments.php" class="btn btn-lp-outline">
                        Cancel
                    </a>
                </div>
            </div>
        </div>

    </div>
</form>

<!-- ═══ Dynamic Invoice Preview + Overpay Detection ═════════════ -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    var invoiceSel    = document.getElementById('invoice_id');
    var amountInput   = document.getElementById('amount');
    var previewBody   = document.getElementById('previewBody');
    var overpayWrap   = document.getElementById('overpayWrap');

    function currencyFmt(n) {
        return '<?php echo CURRENCY_SYMBOL; ?>' + parseFloat(n).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
    }

    function updatePreview() {
        var opt = invoiceSel.options[invoiceSel.selectedIndex];
        if (!opt || opt.value === '0') {
            previewBody.innerHTML = '<div class="text-center text-muted py-2" style="font-size:.85rem">'
                + '<i class="bi bi-receipt d-block mb-1" style="font-size:1.5rem;opacity:.4"></i>'
                + 'Select an invoice to see details.</div>';
            overpayWrap.style.display = 'none';
            return;
        }

        var text    = opt.textContent.trim();
        var balance = parseFloat(opt.getAttribute('data-balance')) || 0;
        var parts   = text.split(' — ');
        var invNum  = parts[0] || '';
        var client  = parts[1] ? parts[1].split(' (Balance')[0] : '';

        previewBody.innerHTML = ''
            + '<div class="d-flex justify-content-between mb-2" style="font-size:.85rem">'
            + '<span class="text-muted">Invoice</span><span class="fw-semibold">' + invNum + '</span></div>'
            + '<div class="d-flex justify-content-between mb-2" style="font-size:.85rem">'
            + '<span class="text-muted">Client</span><span class="fw-semibold">' + client + '</span></div>'
            + '<hr class="my-2" style="border-color:var(--lp-silver)">'
            + '<div class="d-flex justify-content-between" style="font-size:1rem">'
            + '<span class="fw-bold" style="color:var(--lp-amber)">Balance Due</span>'
            + '<span class="fw-bold" style="color:var(--lp-amber)">' + currencyFmt(balance) + '</span></div>';

        /* Auto-fill amount to balance */
        if (!amountInput.value || parseFloat(amountInput.value) === 0) {
            amountInput.value = balance.toFixed(2);
        }

        checkOverpay(balance);
    }

    function checkOverpay(balance) {
        if (typeof balance === 'undefined') {
            var opt = invoiceSel.options[invoiceSel.selectedIndex];
            balance = opt ? (parseFloat(opt.getAttribute('data-balance')) || 0) : 0;
        }
        var amt = parseFloat(amountInput.value) || 0;
        overpayWrap.style.display = (amt > balance && balance > 0) ? 'block' : 'none';
    }

    invoiceSel.addEventListener('change', updatePreview);
    amountInput.addEventListener('input', function() { checkOverpay(); });

    /* Initial state */
    if (invoiceSel.value !== '0') {
        checkOverpay();
    }
});
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
