<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  login.php — User Authentication
 * ═══════════════════════════════════════════════════════════════
 *
 *  Security measures:
 *    1. CSRF token on every form submission
 *    2. Prepared statements — no SQL injection
 *    3. bcrypt password verification (password_verify)
 *    4. Session ID regeneration on successful login
 *    5. Timing-safe comparison via password_verify internals
 *    6. Brute-force delay (1 second on failure)
 *    7. Generic error message — no user-enumeration
 *    8. Inactive account check
 *    9. last_login_at timestamp updated on every login
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

/* ── Start session ────────────────────────────────────────────── */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/* ── If already logged in, go to dashboard ────────────────────── */
if (is_logged_in()) {
    redirect('dashboard.php');
}

$errors = [];
$email  = '';

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Login attempt
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    /* ── CSRF check ────────────────────────────────────────── */
    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        $errors[] = 'Invalid security token. Please try again.';
    }

    /* ── Sanitize input ────────────────────────────────────── */
    $email    = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';

    /* ── Basic validation ──────────────────────────────────── */
    if ($email === '') {
        $errors[] = 'Email address is required.';
    }
    if ($password === '') {
        $errors[] = 'Password is required.';
    }

    /* ── Attempt authentication ────────────────────────────── */
    if (empty($errors)) {

        $user = db_select_one(
            "SELECT id, full_name, email, password_hash, role, client_id, is_active
             FROM users
             WHERE email = ?
             LIMIT 1",
            ['s', $email]
        );

        /*
         * Use a single generic message for both "no such user" and
         * "wrong password" to prevent user-enumeration attacks.
         */
        $generic_fail = 'Invalid email or password.';

        if ($user === null) {
            /* No user found — still run password_verify against a
               dummy hash to keep response time constant. */
            password_verify($password, '$2y$12$xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx');
            sleep(1); // Brute-force delay
            $errors[] = $generic_fail;

        } elseif (!(int)$user['is_active']) {
            /* Account deactivated */
            sleep(1);
            $errors[] = 'This account has been deactivated. Contact your accountant.';

        } elseif (!password_verify($password, $user['password_hash'])) {
            /* Wrong password */
            sleep(1);
            $errors[] = $generic_fail;

        } else {
            /* ── SUCCESS — set up session ──────────────────── */

            /* Regenerate session ID to prevent fixation */
            session_regenerate_id(true);

            $_SESSION['user_id']    = (int) $user['id'];
            $_SESSION['user_name']  = $user['full_name'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_role']  = $user['role'];
            $_SESSION['client_id']  = $user['client_id'] ? (int) $user['client_id'] : null;

            /* Refresh the CSRF token for the new session */
            csrf_regenerate();

            /* Update last_login_at timestamp */
            db_execute(
                "UPDATE users SET last_login_at = NOW() WHERE id = ?",
                ['i', $user['id']]
            );

            /* Redirect to dashboard */
            flash('success', 'Welcome back, ' . $user['full_name'] . '!');
            redirect('dashboard.php');
        }
    }
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER — Login page (no header/footer, standalone layout)
 * ══════════════════════════════════════════════════════════════ */
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=5.0">
    <title>Sign In — <?php echo APP_NAME; ?></title>

    <!-- Bootstrap 5.3 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
          rel="stylesheet"
          integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YcnS/1lM6pRhGFi7gWzm0NjBgL1E9Rm0isG"
          crossorigin="anonymous">

    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css"
          rel="stylesheet">

    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,300..700;1,9..40,300..700&family=Instrument+Serif:ital@0;1&display=swap"
          rel="stylesheet">

    <style>
        :root {
            --lp-font-body:    'DM Sans', system-ui, sans-serif;
            --lp-font-display: 'Instrument Serif', Georgia, serif;
            --lp-navy:         #0f172a;
            --lp-navy-light:   #1e293b;
            --lp-slate:        #334155;
            --lp-slate-mid:    #64748b;
            --lp-slate-light:  #94a3b8;
            --lp-silver:       #e2e8f0;
            --lp-snow:         #f8fafc;
            --lp-white:        #ffffff;
            --lp-emerald:      #059669;
            --lp-emerald-soft: #d1fae5;
            --lp-red:          #dc2626;
            --lp-red-soft:     #fee2e2;
            --lp-blue:         #2563eb;
            --lp-radius:       0.625rem;
            --lp-radius-lg:    1rem;
            --lp-shadow:       0 1px 3px rgba(15,23,42,.08), 0 1px 2px rgba(15,23,42,.04);
            --lp-shadow-lg:    0 10px 15px -3px rgba(15,23,42,.08), 0 4px 6px -4px rgba(15,23,42,.04);
            --lp-shadow-xl:    0 20px 25px -5px rgba(15,23,42,.08), 0 8px 10px -6px rgba(15,23,42,.04);
            --lp-transition:   all .2s cubic-bezier(.4,0,.2,1);
        }

        *, *::before, *::after { box-sizing: border-box; }

        body {
            font-family: var(--lp-font-body);
            margin: 0;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--lp-snow);
            color: var(--lp-slate);
            -webkit-font-smoothing: antialiased;
        }

        /* ── Full page layout: illustration left + form right ── */
        .login-wrapper {
            display: flex;
            width: 100%;
            max-width: 960px;
            min-height: 580px;
            margin: 1.5rem;
            background: var(--lp-white);
            border-radius: var(--lp-radius-lg);
            box-shadow: var(--lp-shadow-xl);
            overflow: hidden;
            animation: fadeUp .5s ease-out;
        }
        @keyframes fadeUp {
            from { opacity: 0; transform: translateY(1rem); }
            to   { opacity: 1; transform: translateY(0); }
        }

        /* ── Left panel: brand + decoration ────────────────── */
        .login-brand-panel {
            flex: 0 0 42%;
            background: var(--lp-navy);
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            padding: 2.5rem;
            position: relative;
            overflow: hidden;
        }
        .login-brand-panel::before {
            content: '';
            position: absolute;
            top: -40%; right: -30%;
            width: 450px; height: 450px;
            border-radius: 50%;
            background: rgba(255,255,255,.04);
        }
        .login-brand-panel::after {
            content: '';
            position: absolute;
            bottom: -20%; left: -20%;
            width: 350px; height: 350px;
            border-radius: 50%;
            background: rgba(255,255,255,.03);
        }
        .brand-top {
            position: relative;
            z-index: 1;
        }
        .brand-logo {
            width: 48px; height: 48px;
            background: rgba(255,255,255,.12);
            border-radius: var(--lp-radius);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--lp-white);
            font-size: 1.35rem;
            margin-bottom: 1.75rem;
        }
        .brand-heading {
            font-family: var(--lp-font-display);
            font-size: 2rem;
            color: var(--lp-white);
            line-height: 1.25;
            margin: 0 0 .75rem;
        }
        .brand-sub {
            font-size: .9rem;
            color: rgba(255,255,255,.55);
            line-height: 1.55;
        }
        .brand-bottom {
            position: relative;
            z-index: 1;
        }
        .brand-features {
            list-style: none;
            padding: 0;
            margin: 0;
            display: flex;
            flex-direction: column;
            gap: .65rem;
        }
        .brand-features li {
            display: flex;
            align-items: center;
            gap: .6rem;
            font-size: .82rem;
            color: rgba(255,255,255,.6);
        }
        .brand-features li i {
            color: var(--lp-emerald);
            font-size: .9rem;
        }

        /* ── Right panel: form ─────────────────────────────── */
        .login-form-panel {
            flex: 1;
            display: flex;
            flex-direction: column;
            justify-content: center;
            padding: 3rem 2.75rem;
        }
        .login-form-panel h2 {
            font-family: var(--lp-font-display);
            font-size: 1.65rem;
            color: var(--lp-navy);
            margin: 0 0 .3rem;
        }
        .login-form-panel .login-subtitle {
            font-size: .875rem;
            color: var(--lp-slate-mid);
            margin-bottom: 1.75rem;
        }

        /* ── Form Elements ─────────────────────────────────── */
        .lp-input-group {
            margin-bottom: 1.25rem;
        }
        .lp-input-group label {
            display: block;
            font-size: .8rem;
            font-weight: 600;
            color: var(--lp-navy);
            margin-bottom: .4rem;
        }
        .lp-input-wrap {
            position: relative;
        }
        .lp-input-wrap i.field-icon {
            position: absolute;
            left: .9rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--lp-slate-light);
            font-size: .95rem;
            pointer-events: none;
            transition: var(--lp-transition);
        }
        .lp-input-wrap input {
            width: 100%;
            padding: .65rem .9rem .65rem 2.6rem;
            font-family: var(--lp-font-body);
            font-size: .875rem;
            color: var(--lp-slate);
            border: 1.5px solid var(--lp-silver);
            border-radius: var(--lp-radius);
            background: var(--lp-white);
            transition: var(--lp-transition);
        }
        .lp-input-wrap input:focus {
            outline: none;
            border-color: var(--lp-navy);
            box-shadow: 0 0 0 3px rgba(15,23,42,.08);
        }
        .lp-input-wrap input:focus + i.field-icon,
        .lp-input-wrap input:focus ~ i.field-icon {
            color: var(--lp-navy);
        }
        .lp-input-wrap input::placeholder {
            color: var(--lp-slate-light);
        }

        /* Password toggle */
        .pw-toggle {
            position: absolute;
            right: .75rem;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--lp-slate-light);
            cursor: pointer;
            padding: .2rem;
            font-size: 1rem;
            transition: var(--lp-transition);
        }
        .pw-toggle:hover { color: var(--lp-navy); }

        /* ── Submit button ─────────────────────────────────── */
        .login-btn {
            width: 100%;
            padding: .7rem;
            font-family: var(--lp-font-body);
            font-size: .9rem;
            font-weight: 600;
            color: var(--lp-white);
            background: var(--lp-navy);
            border: none;
            border-radius: var(--lp-radius);
            cursor: pointer;
            transition: var(--lp-transition);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: .5rem;
            margin-top: .5rem;
        }
        .login-btn:hover {
            background: var(--lp-navy-light);
            transform: translateY(-1px);
            box-shadow: var(--lp-shadow-lg);
        }
        .login-btn:active {
            transform: translateY(0);
        }

        /* ── Error display ─────────────────────────────────── */
        .login-errors {
            background: var(--lp-red-soft);
            border: 1px solid #fecaca;
            border-radius: var(--lp-radius);
            padding: .75rem 1rem;
            margin-bottom: 1.25rem;
            animation: fadeUp .3s ease-out;
        }
        .login-errors ul {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        .login-errors li {
            font-size: .82rem;
            color: var(--lp-red);
            font-weight: 500;
            padding: .15rem 0;
            display: flex;
            align-items: center;
            gap: .4rem;
        }

        /* ── Demo credentials hint ─────────────────────────── */
        .demo-hint {
            margin-top: 1.5rem;
            background: var(--lp-snow);
            border: 1px solid var(--lp-silver);
            border-radius: var(--lp-radius);
            padding: .85rem 1rem;
            font-size: .78rem;
            color: var(--lp-slate-mid);
        }
        .demo-hint strong {
            color: var(--lp-navy);
        }
        .demo-hint code {
            background: var(--lp-white);
            border: 1px solid var(--lp-silver);
            border-radius: 4px;
            padding: .1rem .35rem;
            font-size: .75rem;
            color: var(--lp-slate);
        }

        /* ── Footer line ───────────────────────────────────── */
        .login-footer {
            margin-top: 2rem;
            font-size: .78rem;
            color: var(--lp-slate-light);
            text-align: center;
        }

        /* ── Responsive: collapse brand panel on mobile ────── */
        @media (max-width: 768px) {
            .login-wrapper {
                flex-direction: column;
                max-width: 440px;
                min-height: auto;
            }
            .login-brand-panel {
                flex: none;
                padding: 2rem 2rem 1.5rem;
            }
            .brand-heading { font-size: 1.55rem; }
            .brand-bottom  { display: none; }
            .login-form-panel { padding: 2rem; }
        }

        @media (max-width: 480px) {
            body { align-items: flex-start; padding-top: .5rem; }
            .login-wrapper { margin: .75rem; }
            .login-form-panel { padding: 1.5rem 1.25rem; }
            .brand-sub { display: none; }
        }
    </style>
</head>
<body>

<div class="login-wrapper">

    <!-- ═══ Left: Brand Panel ════════════════════════════════ -->
    <div class="login-brand-panel">
        <div class="brand-top">
            <div class="brand-logo">
                <i class="bi bi-journal-bookmark-fill"></i>
            </div>
            <h1 class="brand-heading"><?php echo APP_NAME; ?></h1>
            <p class="brand-sub">
                Professional accounting platform for managing clients,
                invoices, expenses, and financial reporting — all in one place.
            </p>
        </div>
        <div class="brand-bottom">
            <ul class="brand-features">
                <li><i class="bi bi-check-circle-fill"></i> Client & invoice management</li>
                <li><i class="bi bi-check-circle-fill"></i> Income & expense tracking</li>
                <li><i class="bi bi-check-circle-fill"></i> Payment recording & history</li>
                <li><i class="bi bi-check-circle-fill"></i> Financial reports & analytics</li>
                <li><i class="bi bi-check-circle-fill"></i> Secure client portal access</li>
            </ul>
        </div>
    </div>

    <!-- ═══ Right: Login Form ════════════════════════════════ -->
    <div class="login-form-panel">

        <h2>Welcome back</h2>
        <p class="login-subtitle">Sign in to your account to continue.</p>

        <!-- Errors -->
        <?php if (!empty($errors)): ?>
        <div class="login-errors">
            <ul>
                <?php foreach ($errors as $err): ?>
                <li><i class="bi bi-exclamation-circle-fill"></i> <?php echo h($err); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>

        <!-- Form -->
        <form method="POST" action="login.php" autocomplete="on" novalidate>
            <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
                   value="<?php echo csrf_token(); ?>">

            <!-- Email -->
            <div class="lp-input-group">
                <label for="email">Email Address</label>
                <div class="lp-input-wrap">
                    <input type="email"
                           id="email"
                           name="email"
                           value="<?php echo h($email); ?>"
                           placeholder="you@company.com"
                           autocomplete="email"
                           required>
                    <i class="bi bi-envelope field-icon"></i>
                </div>
            </div>

            <!-- Password -->
            <div class="lp-input-group">
                <label for="password">Password</label>
                <div class="lp-input-wrap">
                    <input type="password"
                           id="password"
                           name="password"
                           placeholder="Enter your password"
                           autocomplete="current-password"
                           required>
                    <i class="bi bi-lock field-icon"></i>
                    <button type="button" class="pw-toggle" aria-label="Toggle password visibility"
                            onclick="togglePw()">
                        <i class="bi bi-eye" id="pw-icon"></i>
                    </button>
                </div>
            </div>

            <!-- Submit -->
            <button type="submit" class="login-btn">
                <i class="bi bi-box-arrow-in-right"></i>
                Sign In
            </button>
        </form>

        <!-- Demo credentials -->
        <div class="demo-hint">
            <strong>Demo Credentials</strong><br>
            Email: <code>admin@ledgerpro.com</code><br>
            Password: <code>Admin@1234</code>
        </div>

        <div class="login-footer">
            &copy; <?php echo date('Y'); ?> <?php echo APP_NAME; ?> &middot; v<?php echo APP_VERSION; ?>
        </div>

    </div>
</div>

<!-- ── Password Toggle Script ──────────────────────────────── -->
<script>
function togglePw() {
    var input = document.getElementById('password');
    var icon  = document.getElementById('pw-icon');
    if (input.type === 'password') {
        input.type = 'text';
        icon.classList.replace('bi-eye', 'bi-eye-slash');
    } else {
        input.type = 'password';
        icon.classList.replace('bi-eye-slash', 'bi-eye');
    }
}

/* Auto-focus email or password field */
document.addEventListener('DOMContentLoaded', function() {
    var emailField = document.getElementById('email');
    if (emailField.value.length > 0) {
        document.getElementById('password').focus();
    } else {
        emailField.focus();
    }
});
</script>

</body>
</html>
