<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  invoice_form.php — Create / Edit Invoice
 * ═══════════════════════════════════════════════════════════════
 *
 *  Dual-mode:
 *    GET  /invoice_form.php                → new invoice
 *    GET  /invoice_form.php?id=5           → edit invoice #5
 *    GET  /invoice_form.php?client_id=3    → new invoice pre-filled for client 3
 *    POST                                  → validate + save (INSERT or UPDATE)
 *
 *  Line items:
 *    - Dynamic JS rows: add / remove without page reload
 *    - Live calculation: qty × price = line total → subtotal → tax → grand total
 *    - Submitted as arrays: items[desc][], items[qty][], items[price][]
 *
 *  Save logic:
 *    - Wrapped in a DB transaction
 *    - On edit: deletes old line items, re-inserts new ones
 *    - Recalculates subtotal, tax_amount, total, balance_due
 *
 *  Security:
 *    - Accountant only
 *    - CSRF on every submission
 *    - Prepared statements throughout
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ══════════════════════════════════════════════════════════════
 *  MODE DETECTION
 * ══════════════════════════════════════════════════════════════ */
$edit_id = isset($_GET['id']) ? (int) $_GET['id'] : 0;
$is_edit = $edit_id > 0;
$errors  = [];

/* ── Default field values ─────────────────────────────────────── */
$f = [
    'client_id'       => (int) ($_GET['client_id'] ?? 0),
    'invoice_number'  => '',
    'status'          => STATUS_DRAFT,
    'issue_date'      => date('Y-m-d'),
    'due_date'        => date('Y-m-d', strtotime('+30 days')),
    'tax_rate'        => '0.00',
    'discount_amount' => '0.00',
    'notes'           => '',
    'terms'           => 'Payment is due within 30 days of the invoice date.',
];

/* Line items: arrays of description, quantity, unit_price */
$items = [
    ['description' => '', 'quantity' => '1.00', 'unit_price' => '0.00']
];

/* ── Auto-generate next invoice number for new invoices ───────── */
if (!$is_edit && $f['invoice_number'] === '') {
    $last_num = db_scalar("SELECT invoice_number FROM invoices ORDER BY id DESC LIMIT 1");
    if ($last_num && preg_match('/INV-(\d+)/', $last_num, $m)) {
        $f['invoice_number'] = 'INV-' . str_pad((int)$m[1] + 1, 5, '0', STR_PAD_LEFT);
    } else {
        $f['invoice_number'] = 'INV-00001';
    }
}

/* ── If editing, load existing invoice + line items ───────────── */
if ($is_edit) {
    $invoice = db_select_one("SELECT * FROM invoices WHERE id = ?", ['i', $edit_id]);
    if (!$invoice) {
        flash('error', 'Invoice not found.');
        redirect('invoices.php');
    }

    $f['client_id']       = (int) $invoice['client_id'];
    $f['invoice_number']  = $invoice['invoice_number'];
    $f['status']          = $invoice['status'];
    $f['issue_date']      = $invoice['issue_date'];
    $f['due_date']        = $invoice['due_date'];
    $f['tax_rate']        = $invoice['tax_rate'];
    $f['discount_amount'] = $invoice['discount_amount'];
    $f['notes']           = $invoice['notes'] ?? '';
    $f['terms']           = $invoice['terms'] ?? '';

    $db_items = db_select(
        "SELECT description, quantity, unit_price
         FROM invoice_items WHERE invoice_id = ? ORDER BY sort_order ASC",
        ['i', $edit_id]
    );
    if (!empty($db_items)) {
        $items = $db_items;
    }
}

/* ── Client list for dropdown ─────────────────────────────────── */
$client_list = db_select(
    "SELECT id, company_name FROM clients WHERE is_active = 1 ORDER BY company_name ASC"
);

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Validate & Save
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    /* ── CSRF ──────────────────────────────────────────────── */
    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        $errors[] = 'Invalid security token. Please try again.';
    }

    /* ── Collect fields ────────────────────────────────────── */
    $f['client_id']       = (int) ($_POST['client_id'] ?? 0);
    $f['invoice_number']  = trim($_POST['invoice_number'] ?? '');
    $f['status']          = trim($_POST['status'] ?? STATUS_DRAFT);
    $f['issue_date']      = trim($_POST['issue_date'] ?? '');
    $f['due_date']        = trim($_POST['due_date'] ?? '');
    $f['tax_rate']        = trim($_POST['tax_rate'] ?? '0.00');
    $f['discount_amount'] = trim($_POST['discount_amount'] ?? '0.00');
    $f['notes']           = trim($_POST['notes'] ?? '');
    $f['terms']           = trim($_POST['terms'] ?? '');

    /* ── Collect line items from POST arrays ───────────────── */
    $post_desc  = $_POST['items']['desc']  ?? [];
    $post_qty   = $_POST['items']['qty']   ?? [];
    $post_price = $_POST['items']['price'] ?? [];

    $items = [];
    $item_count = count($post_desc);
    for ($i = 0; $i < $item_count; $i++) {
        $d = trim($post_desc[$i] ?? '');
        $q = trim($post_qty[$i] ?? '1');
        $p = trim($post_price[$i] ?? '0');
        /* Skip completely blank rows */
        if ($d === '' && (float)$q === 0.0 && (float)$p === 0.0) continue;
        $items[] = [
            'description' => $d,
            'quantity'    => $q,
            'unit_price'  => $p,
        ];
    }

    /* ── Validation ────────────────────────────────────────── */
    if ($f['client_id'] <= 0) {
        $errors[] = 'Please select a client.';
    }

    if ($f['invoice_number'] === '') {
        $errors[] = 'Invoice number is required.';
    } else {
        /* Unique check (exclude self on edit) */
        $dup_sql = "SELECT id FROM invoices WHERE invoice_number = ?";
        $dup_p   = ['s', $f['invoice_number']];
        if ($is_edit) {
            $dup_sql .= " AND id != ?";
            $dup_p[0] .= 'i';
            $dup_p[]   = $edit_id;
        }
        if (db_select_one($dup_sql, $dup_p)) {
            $errors[] = 'This invoice number is already in use.';
        }
    }

    $valid_statuses = ['draft','sent','paid','overdue','cancelled'];
    if (!in_array($f['status'], $valid_statuses)) {
        $errors[] = 'Invalid status selected.';
    }

    if ($f['issue_date'] === '' || !strtotime($f['issue_date'])) {
        $errors[] = 'A valid issue date is required.';
    }
    if ($f['due_date'] === '' || !strtotime($f['due_date'])) {
        $errors[] = 'A valid due date is required.';
    }
    if ($f['issue_date'] && $f['due_date'] && $f['due_date'] < $f['issue_date']) {
        $errors[] = 'Due date cannot be before the issue date.';
    }

    if (!is_numeric($f['tax_rate']) || (float)$f['tax_rate'] < 0 || (float)$f['tax_rate'] > 100) {
        $errors[] = 'Tax rate must be between 0 and 100.';
    }

    if (!is_numeric($f['discount_amount']) || (float)$f['discount_amount'] < 0) {
        $errors[] = 'Discount must be zero or positive.';
    }

    /* Line item validation */
    if (empty($items)) {
        $errors[] = 'At least one line item is required.';
    } else {
        foreach ($items as $idx => $itm) {
            $row = $idx + 1;
            if (trim($itm['description']) === '') {
                $errors[] = "Line item #{$row}: description is required.";
            }
            if (!is_numeric($itm['quantity']) || (float)$itm['quantity'] <= 0) {
                $errors[] = "Line item #{$row}: quantity must be greater than zero.";
            }
            if (!is_numeric($itm['unit_price']) || (float)$itm['unit_price'] < 0) {
                $errors[] = "Line item #{$row}: unit price must be zero or positive.";
            }
        }
    }

    /* ── Save (inside transaction) ─────────────────────────── */
    if (empty($errors)) {

        /* Calculate totals */
        $subtotal = 0;
        foreach ($items as $itm) {
            $subtotal += (float)$itm['quantity'] * (float)$itm['unit_price'];
        }
        $tax_rate     = (float) $f['tax_rate'];
        $tax_amount   = round($subtotal * ($tax_rate / 100), 2);
        $discount     = (float) $f['discount_amount'];
        $total        = round($subtotal + $tax_amount - $discount, 2);

        /* For edit: figure existing amount_paid to update balance */
        $amount_paid = 0;
        if ($is_edit) {
            $amount_paid = (float) db_scalar(
                "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE invoice_id = ?",
                ['i', $edit_id]
            );
        }
        $balance_due = round($total - $amount_paid, 2);

        db_begin();
        try {
            if ($is_edit) {
                db_execute(
                    "UPDATE invoices SET
                        client_id = ?, invoice_number = ?, status = ?,
                        issue_date = ?, due_date = ?,
                        subtotal = ?, tax_rate = ?, tax_amount = ?,
                        discount_amount = ?, total = ?,
                        amount_paid = ?, balance_due = ?,
                        notes = ?, terms = ?
                     WHERE id = ?",
                    [
                        'issssdddddddssi',
                        $f['client_id'], $f['invoice_number'], $f['status'],
                        $f['issue_date'], $f['due_date'],
                        $subtotal, $tax_rate, $tax_amount,
                        $discount, $total,
                        $amount_paid, $balance_due,
                        $f['notes'], $f['terms'],
                        $edit_id
                    ]
                );

                /* Delete old line items (will re-insert below) */
                db_execute("DELETE FROM invoice_items WHERE invoice_id = ?", ['i', $edit_id]);
                $invoice_id = $edit_id;

            } else {
                $invoice_id = db_insert(
                    "INSERT INTO invoices
                        (client_id, invoice_number, status, issue_date, due_date,
                         subtotal, tax_rate, tax_amount, discount_amount, total,
                         amount_paid, balance_due, notes, terms, created_by)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, ?, ?, ?, ?)",
                    [
                        'issssddddddssi',
                        $f['client_id'], $f['invoice_number'], $f['status'],
                        $f['issue_date'], $f['due_date'],
                        $subtotal, $tax_rate, $tax_amount,
                        $discount, $total,
                        $total, // balance_due = total for new invoices
                        $f['notes'], $f['terms'],
                        $_SESSION['user_id']
                    ]
                );
            }

            /* Insert line items */
            foreach ($items as $sort => $itm) {
                $line_total = round((float)$itm['quantity'] * (float)$itm['unit_price'], 2);
                db_insert(
                    "INSERT INTO invoice_items
                        (invoice_id, description, quantity, unit_price, line_total, sort_order)
                     VALUES (?, ?, ?, ?, ?, ?)",
                    [
                        'isdddi',
                        $invoice_id,
                        trim($itm['description']),
                        (float)$itm['quantity'],
                        (float)$itm['unit_price'],
                        $line_total,
                        $sort
                    ]
                );
            }

            db_commit();
            csrf_regenerate();
            flash('success', $is_edit ? 'Invoice updated successfully.' : 'Invoice created successfully.');
            redirect('invoice_view.php?id=' . $invoice_id);

        } catch (Exception $e) {
            db_rollback();
            $errors[] = 'Database error: ' . $e->getMessage();
        }
    }

    /* Ensure at least one blank item row if all were removed */
    if (empty($items)) {
        $items = [['description' => '', 'quantity' => '1.00', 'unit_price' => '0.00']];
    }
}

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = $is_edit ? 'Edit Invoice' : 'New Invoice';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1>
            <i class="bi bi-<?php echo $is_edit ? 'pencil-square' : 'receipt'; ?> me-2" style="font-size:1.4rem"></i>
            <?php echo $is_edit ? 'Edit Invoice' : 'New Invoice'; ?>
        </h1>
        <div class="lp-subtitle">
            <?php echo $is_edit
                ? 'Editing ' . h($f['invoice_number'])
                : 'Create a new invoice for a client.'; ?>
        </div>
    </div>
    <a href="<?php echo APP_URL; ?>/invoices.php" class="btn btn-lp-outline">
        <i class="bi bi-arrow-left me-1"></i> Back to Invoices
    </a>
</div>

<!-- ═══ Errors ══════════════════════════════════════════════════ -->
<?php if (!empty($errors)): ?>
<div class="lp-flash error" style="animation:none;">
    <div>
        <i class="bi bi-exclamation-triangle-fill me-1"></i>
        <strong>Please correct the following:</strong>
        <ul class="mb-0 mt-1 ps-3" style="font-weight:400">
            <?php foreach ($errors as $err): ?>
            <li><?php echo h($err); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
</div>
<?php endif; ?>

<!-- ═══ Invoice Form ════════════════════════════════════════════ -->
<form method="POST"
      action="invoice_form.php<?php echo $is_edit ? '?id=' . $edit_id : ''; ?>"
      class="lp-form" id="invoiceForm" novalidate>

    <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>"
           value="<?php echo csrf_token(); ?>">

    <div class="row g-4">

        <!-- ═══ Left Column: Main Content ═══════════════════════ -->
        <div class="col-lg-8">

            <!-- ── Invoice Details Card ────────────────────────── -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-file-earmark-text me-1"></i> Invoice Details
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <!-- Client -->
                        <div class="col-md-6">
                            <label for="client_id" class="form-label">
                                Client <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" id="client_id" name="client_id" required>
                                <option value="0">— Select client —</option>
                                <?php foreach ($client_list as $cl): ?>
                                <option value="<?php echo (int)$cl['id']; ?>"
                                    <?php echo $f['client_id'] === (int)$cl['id'] ? 'selected' : ''; ?>>
                                    <?php echo h($cl['company_name']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Invoice number -->
                        <div class="col-md-6">
                            <label for="invoice_number" class="form-label">
                                Invoice Number <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" id="invoice_number"
                                   name="invoice_number"
                                   value="<?php echo h($f['invoice_number']); ?>"
                                   required maxlength="30">
                        </div>

                        <!-- Issue date -->
                        <div class="col-md-4">
                            <label for="issue_date" class="form-label">
                                Issue Date <span class="text-danger">*</span>
                            </label>
                            <input type="date" class="form-control" id="issue_date"
                                   name="issue_date"
                                   value="<?php echo h($f['issue_date']); ?>" required>
                        </div>

                        <!-- Due date -->
                        <div class="col-md-4">
                            <label for="due_date" class="form-label">
                                Due Date <span class="text-danger">*</span>
                            </label>
                            <input type="date" class="form-control" id="due_date"
                                   name="due_date"
                                   value="<?php echo h($f['due_date']); ?>" required>
                        </div>

                        <!-- Status -->
                        <div class="col-md-4">
                            <label for="status" class="form-label">Status</label>
                            <select class="form-select" id="status" name="status">
                                <?php
                                $status_options = [
                                    'draft'     => 'Draft',
                                    'sent'      => 'Sent',
                                    'paid'      => 'Paid',
                                    'overdue'   => 'Overdue',
                                    'cancelled' => 'Cancelled',
                                ];
                                foreach ($status_options as $val => $label):
                                ?>
                                <option value="<?php echo $val; ?>"
                                    <?php echo $f['status'] === $val ? 'selected' : ''; ?>>
                                    <?php echo $label; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>
            </div>

            <!-- ── Line Items Card ─────────────────────────────── -->
            <div class="lp-card mb-4">
                <div class="card-header d-flex align-items-center justify-content-between">
                    <span><i class="bi bi-list-ul me-1"></i> Line Items</span>
                    <button type="button" class="btn btn-sm btn-lp-primary" id="addLineBtn"
                            style="font-size:.78rem; padding:.25rem .7rem">
                        <i class="bi bi-plus-lg me-1"></i> Add Row
                    </button>
                </div>
                <div class="card-body p-0">
                    <div class="lp-table-responsive">
                        <table class="lp-table" id="lineItemsTable" style="min-width:560px">
                            <thead>
                                <tr>
                                    <th style="width:45%">Description <span class="text-danger">*</span></th>
                                    <th style="width:14%" class="text-center">Qty</th>
                                    <th style="width:18%" class="text-end">Unit Price</th>
                                    <th style="width:16%" class="text-end">Total</th>
                                    <th style="width:7%" class="text-center"></th>
                                </tr>
                            </thead>
                            <tbody id="lineItemsBody">
                                <?php foreach ($items as $idx => $itm): ?>
                                <tr class="lp-line-item-row">
                                    <td>
                                        <input type="text" class="form-control form-control-sm"
                                               name="items[desc][]"
                                               value="<?php echo h($itm['description']); ?>"
                                               placeholder="Service or product description"
                                               required>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-center"
                                               name="items[qty][]" data-calc="qty"
                                               value="<?php echo h($itm['quantity']); ?>"
                                               step="0.01" min="0.01" required>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control form-control-sm text-end"
                                               name="items[price][]" data-calc="price"
                                               value="<?php echo h($itm['unit_price']); ?>"
                                               step="0.01" min="0" required>
                                    </td>
                                    <td class="text-end fw-semibold" style="vertical-align:middle; color:var(--lp-navy)">
                                        <span data-calc="line-total">0.00</span>
                                    </td>
                                    <td class="text-center">
                                        <button type="button" class="btn btn-sm text-danger border-0 remove-line-btn"
                                                title="Remove row" style="padding:.2rem .45rem">
                                            <i class="bi bi-x-lg"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Totals -->
                    <div class="px-3 py-3" style="border-top:2px solid var(--lp-silver); background:var(--lp-snow)">
                        <div class="row justify-content-end">
                            <div class="col-sm-7 col-md-5 col-lg-5">
                                <!-- Subtotal -->
                                <div class="d-flex justify-content-between mb-2" style="font-size:.88rem">
                                    <span class="text-muted">Subtotal</span>
                                    <span class="fw-semibold" style="color:var(--lp-navy)">
                                        <?php echo CURRENCY_SYMBOL; ?><span id="calc-subtotal">0.00</span>
                                    </span>
                                </div>

                                <!-- Tax rate -->
                                <div class="d-flex justify-content-between align-items-center mb-2" style="font-size:.88rem">
                                    <div class="d-flex align-items-center gap-2">
                                        <span class="text-muted">Tax</span>
                                        <div class="input-group input-group-sm" style="width:100px">
                                            <input type="number" class="form-control form-control-sm text-end"
                                                   id="tax_rate" name="tax_rate"
                                                   value="<?php echo h($f['tax_rate']); ?>"
                                                   step="0.01" min="0" max="100">
                                            <span class="input-group-text" style="font-size:.78rem">%</span>
                                        </div>
                                    </div>
                                    <span class="fw-semibold" style="color:var(--lp-navy)">
                                        <?php echo CURRENCY_SYMBOL; ?><span id="calc-tax">0.00</span>
                                    </span>
                                </div>

                                <!-- Discount -->
                                <div class="d-flex justify-content-between align-items-center mb-3" style="font-size:.88rem">
                                    <div class="d-flex align-items-center gap-2">
                                        <span class="text-muted">Discount</span>
                                        <div class="input-group input-group-sm" style="width:120px">
                                            <span class="input-group-text" style="font-size:.78rem"><?php echo CURRENCY_SYMBOL; ?></span>
                                            <input type="number" class="form-control form-control-sm text-end"
                                                   id="discount_amount" name="discount_amount"
                                                   value="<?php echo h($f['discount_amount']); ?>"
                                                   step="0.01" min="0">
                                        </div>
                                    </div>
                                    <span class="text-danger fw-semibold">
                                        -<?php echo CURRENCY_SYMBOL; ?><span id="calc-discount">0.00</span>
                                    </span>
                                </div>

                                <!-- Grand total -->
                                <div class="d-flex justify-content-between pt-2"
                                     style="border-top:2px solid var(--lp-navy); font-size:1.05rem">
                                    <span class="fw-bold" style="color:var(--lp-navy)">Total</span>
                                    <span class="fw-bold" style="color:var(--lp-navy)">
                                        <?php echo CURRENCY_SYMBOL; ?><span id="calc-total">0.00</span>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- ── Notes & Terms ───────────────────────────────── -->
            <div class="lp-card mb-4">
                <div class="card-header">
                    <i class="bi bi-chat-left-text me-1"></i> Notes & Terms
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="notes" name="notes" rows="3"
                                      style="resize:vertical"
                                      placeholder="Notes visible to the client on the invoice…"
                            ><?php echo h($f['notes']); ?></textarea>
                        </div>
                        <div class="col-md-6">
                            <label for="terms" class="form-label">Terms & Conditions</label>
                            <textarea class="form-control" id="terms" name="terms" rows="3"
                                      style="resize:vertical"
                                      placeholder="Payment terms, late fees, etc."
                            ><?php echo h($f['terms']); ?></textarea>
                        </div>
                    </div>
                </div>
            </div>

        </div><!-- /left col -->

        <!-- ═══ Right Column: Sidebar ═══════════════════════════ -->
        <div class="col-lg-4">

            <!-- Summary Card -->
            <div class="lp-card mb-4" style="position:sticky; top:80px">
                <div class="card-header">
                    <i class="bi bi-calculator me-1"></i> Invoice Summary
                </div>
                <div class="card-body">
                    <!-- Live preview numbers -->
                    <div class="mb-3 pb-3" style="border-bottom:1px solid var(--lp-silver)">
                        <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                            <span class="text-muted">Items</span>
                            <span class="fw-semibold" id="sidebar-item-count"><?php echo count($items); ?></span>
                        </div>
                        <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                            <span class="text-muted">Subtotal</span>
                            <span class="fw-semibold"><?php echo CURRENCY_SYMBOL; ?><span id="sidebar-subtotal">0.00</span></span>
                        </div>
                        <div class="d-flex justify-content-between mb-2" style="font-size:.85rem">
                            <span class="text-muted">Tax</span>
                            <span><?php echo CURRENCY_SYMBOL; ?><span id="sidebar-tax">0.00</span></span>
                        </div>
                        <div class="d-flex justify-content-between" style="font-size:.85rem">
                            <span class="text-muted">Discount</span>
                            <span class="text-danger">-<?php echo CURRENCY_SYMBOL; ?><span id="sidebar-discount">0.00</span></span>
                        </div>
                    </div>
                    <div class="d-flex justify-content-between" style="font-size:1.15rem">
                        <span class="fw-bold" style="color:var(--lp-navy)">Total</span>
                        <span class="fw-bold" style="color:var(--lp-navy)">
                            <?php echo CURRENCY_SYMBOL; ?><span id="sidebar-total">0.00</span>
                        </span>
                    </div>
                </div>

                <!-- Action buttons -->
                <div class="card-body border-top d-grid gap-2" style="border-color:var(--lp-silver) !important">
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i>
                        <?php echo $is_edit ? 'Update Invoice' : 'Create Invoice'; ?>
                    </button>
                    <a href="<?php echo APP_URL; ?>/invoices.php" class="btn btn-lp-outline">
                        Cancel
                    </a>
                    <?php if ($is_edit): ?>
                    <a href="<?php echo APP_URL; ?>/invoice_view.php?id=<?php echo $edit_id; ?>"
                       class="btn btn-lp-outline">
                        <i class="bi bi-eye me-1"></i> View Invoice
                    </a>
                    <?php endif; ?>
                </div>
            </div>

        </div><!-- /sidebar -->
    </div><!-- /row -->
</form>

<!-- ═══ Line Item Template (hidden, cloned by JS) ═══════════════ -->
<template id="lineItemTemplate">
    <tr class="lp-line-item-row">
        <td>
            <input type="text" class="form-control form-control-sm"
                   name="items[desc][]" placeholder="Service or product description" required>
        </td>
        <td>
            <input type="number" class="form-control form-control-sm text-center"
                   name="items[qty][]" data-calc="qty" value="1.00" step="0.01" min="0.01" required>
        </td>
        <td>
            <input type="number" class="form-control form-control-sm text-end"
                   name="items[price][]" data-calc="price" value="0.00" step="0.01" min="0" required>
        </td>
        <td class="text-end fw-semibold" style="vertical-align:middle; color:var(--lp-navy)">
            <span data-calc="line-total">0.00</span>
        </td>
        <td class="text-center">
            <button type="button" class="btn btn-sm text-danger border-0 remove-line-btn"
                    title="Remove row" style="padding:.2rem .45rem">
                <i class="bi bi-x-lg"></i>
            </button>
        </td>
    </tr>
</template>

<!-- ═══ Line Item JS ════════════════════════════════════════════ -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    var tbody    = document.getElementById('lineItemsBody');
    var template = document.getElementById('lineItemTemplate');
    var addBtn   = document.getElementById('addLineBtn');

    /* ── Add a new blank row ───────────────────────────────── */
    addBtn.addEventListener('click', function() {
        var clone = template.content.cloneNode(true);
        tbody.appendChild(clone);
        attachRowListeners();
        recalcAll();
        /* Focus the new description field */
        var rows = tbody.querySelectorAll('.lp-line-item-row');
        var last = rows[rows.length - 1];
        last.querySelector('input[name="items[desc][]"]').focus();
    });

    /* ── Remove row (delegated) ────────────────────────────── */
    tbody.addEventListener('click', function(e) {
        var btn = e.target.closest('.remove-line-btn');
        if (!btn) return;
        var rows = tbody.querySelectorAll('.lp-line-item-row');
        if (rows.length <= 1) {
            alert('An invoice must have at least one line item.');
            return;
        }
        btn.closest('tr').remove();
        recalcAll();
    });

    /* ── Recalculate everything ─────────────────────────────── */
    function recalcAll() {
        var subtotal = 0;
        tbody.querySelectorAll('.lp-line-item-row').forEach(function(row) {
            var qty   = parseFloat(row.querySelector('[data-calc="qty"]').value)   || 0;
            var price = parseFloat(row.querySelector('[data-calc="price"]').value) || 0;
            var total = qty * price;
            subtotal += total;
            row.querySelector('[data-calc="line-total"]').textContent = total.toFixed(2);
        });

        var taxRate  = parseFloat(document.getElementById('tax_rate').value) || 0;
        var discount = parseFloat(document.getElementById('discount_amount').value) || 0;
        var taxAmt   = subtotal * (taxRate / 100);
        var grand    = subtotal + taxAmt - discount;

        /* Main totals */
        document.getElementById('calc-subtotal').textContent = subtotal.toFixed(2);
        document.getElementById('calc-tax').textContent      = taxAmt.toFixed(2);
        document.getElementById('calc-discount').textContent = discount.toFixed(2);
        document.getElementById('calc-total').textContent    = grand.toFixed(2);

        /* Sidebar mirror */
        document.getElementById('sidebar-subtotal').textContent  = subtotal.toFixed(2);
        document.getElementById('sidebar-tax').textContent       = taxAmt.toFixed(2);
        document.getElementById('sidebar-discount').textContent  = discount.toFixed(2);
        document.getElementById('sidebar-total').textContent     = grand.toFixed(2);
        document.getElementById('sidebar-item-count').textContent =
            tbody.querySelectorAll('.lp-line-item-row').length;
    }

    /* ── Attach input listeners to all current rows ────────── */
    function attachRowListeners() {
        tbody.querySelectorAll('[data-calc="qty"], [data-calc="price"]').forEach(function(input) {
            /* Remove existing to avoid duplicates */
            input.removeEventListener('input', recalcAll);
            input.addEventListener('input', recalcAll);
        });
    }

    /* ── Tax & discount listeners ──────────────────────────── */
    document.getElementById('tax_rate').addEventListener('input', recalcAll);
    document.getElementById('discount_amount').addEventListener('input', recalcAll);

    /* ── Initial setup ─────────────────────────────────────── */
    attachRowListeners();
    recalcAll();
});
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
