<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  expenses.php — Expense Tracking
 * ═══════════════════════════════════════════════════════════════
 *
 *  Features:
 *    - Stat cards: total expenses, this month, entry count
 *    - Search by description, vendor, or reference
 *    - Filter by category, billable flag, date range
 *    - Create / Edit / Delete with CSRF
 *    - Billable toggle for client-chargeable expenses
 *    - Desktop table + mobile cards
 *
 *  Authorization: Accountant only
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

if (session_status() === PHP_SESSION_NONE) session_start();
require_accountant();

/* ── Expense categories (from config or default) ──────────────── */
$categories = defined('EXPENSE_CATEGORIES')
    ? json_decode(EXPENSE_CATEGORIES, true)
    : ['Rent','Utilities','Salaries','Office Supplies','Travel','Insurance',
       'Marketing','Software','Professional Fees','Taxes','Depreciation',
       'Maintenance','Miscellaneous'];

/* ══════════════════════════════════════════════════════════════
 *  HANDLE POST — Create / Edit / Delete
 * ══════════════════════════════════════════════════════════════ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (!csrf_validate($_POST[CSRF_TOKEN_NAME] ?? '')) {
        flash('error', 'Invalid security token.');
        redirect('expenses.php');
    }

    $action = $_POST['action'] ?? '';

    /* ── DELETE ─────────────────────────────────────────────── */
    if ($action === 'delete') {
        $del_id = (int) ($_POST['expense_id'] ?? 0);
        if ($del_id > 0) {
            db_execute("DELETE FROM expenses WHERE id = ?", ['i', $del_id]);
            flash('success', 'Expense deleted.');
        }
        csrf_regenerate();
        redirect('expenses.php');
    }

    /* ── CREATE / UPDATE ───────────────────────────────────── */
    if ($action === 'save') {
        $edit_id      = (int) ($_POST['edit_id'] ?? 0);
        $client_id    = (int) ($_POST['client_id'] ?? 0);
        $category     = trim($_POST['category'] ?? '');
        $description  = trim($_POST['description'] ?? '');
        $amount       = trim($_POST['amount'] ?? '');
        $expense_date = trim($_POST['expense_date'] ?? '');
        $vendor       = trim($_POST['vendor'] ?? '');
        $reference    = trim($_POST['reference'] ?? '');
        $is_billable  = isset($_POST['is_billable']) ? 1 : 0;
        $notes        = trim($_POST['notes'] ?? '');

        $errors = [];
        if ($category === '')     $errors[] = 'Category is required.';
        if ($description === '')  $errors[] = 'Description is required.';
        if (!is_numeric($amount) || (float)$amount <= 0) $errors[] = 'A valid positive amount is required.';
        if ($expense_date === '' || !strtotime($expense_date)) $errors[] = 'A valid date is required.';

        if (!empty($errors)) {
            flash('error', implode(' ', $errors));
        } else {
            $client_val = $client_id > 0 ? $client_id : null;

            if ($edit_id > 0) {
                db_execute(
                    "UPDATE expenses SET client_id = ?, category = ?, description = ?,
                        amount = ?, expense_date = ?, vendor = ?, reference = ?,
                        is_billable = ?, notes = ?
                     WHERE id = ?",
                    ['issdsssisi',
                     $client_val, $category, $description,
                     (float)$amount, $expense_date, $vendor, $reference,
                     $is_billable, $notes,
                     $edit_id]
                );
                flash('success', 'Expense updated.');
            } else {
                db_insert(
                    "INSERT INTO expenses
                        (client_id, category, description, amount, expense_date,
                         vendor, reference, is_billable, notes, recorded_by)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                    ['issdsssisi',
                     $client_val, $category, $description,
                     (float)$amount, $expense_date, $vendor, $reference,
                     $is_billable, $notes,
                     $_SESSION['user_id']]
                );
                flash('success', 'Expense created.');
            }
        }
        csrf_regenerate();
        redirect('expenses.php');
    }
}

/* ══════════════════════════════════════════════════════════════
 *  FILTERS
 * ══════════════════════════════════════════════════════════════ */
$search     = trim($_GET['search'] ?? '');
$cat_filter = trim($_GET['category'] ?? 'all');
$bill_filter = $_GET['billable'] ?? 'all';
$date_from  = trim($_GET['date_from'] ?? '');
$date_to    = trim($_GET['date_to'] ?? '');

$where  = [];
$params = [];
$types  = '';

if ($search !== '') {
    $safe = db_escape_like($search);
    $where[]  = "(e.description LIKE CONCAT('%',?,'%') OR e.vendor LIKE CONCAT('%',?,'%') OR e.reference LIKE CONCAT('%',?,'%'))";
    $params[] = $safe; $params[] = $safe; $params[] = $safe;
    $types   .= 'sss';
}
if ($cat_filter !== 'all' && $cat_filter !== '') {
    $where[] = "e.category = ?"; $params[] = $cat_filter; $types .= 's';
}
if ($bill_filter === 'yes') {
    $where[] = "e.is_billable = 1";
} elseif ($bill_filter === 'no') {
    $where[] = "e.is_billable = 0";
}
if ($date_from !== '') { $where[] = "e.expense_date >= ?"; $params[] = $date_from; $types .= 's'; }
if ($date_to !== '')   { $where[] = "e.expense_date <= ?"; $params[] = $date_to;   $types .= 's'; }

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

/* ── Stats ────────────────────────────────────────────────────── */
$sp = empty($types) ? [] : array_merge([$types], $params);
$stat_total = (float) db_scalar("SELECT COALESCE(SUM(e.amount),0) FROM expenses e {$where_sql}", $sp);
$stat_count = (int)   db_scalar("SELECT COUNT(*) FROM expenses e {$where_sql}", $sp);
$stat_month = (float) db_scalar(
    "SELECT COALESCE(SUM(amount),0) FROM expenses WHERE expense_date >= ?",
    ['s', date('Y-m-01')]
);
$stat_billable = (float) db_scalar(
    "SELECT COALESCE(SUM(amount),0) FROM expenses WHERE is_billable = 1"
);

/* ── Pagination ───────────────────────────────────────────────── */
$pager = db_paginate("SELECT COUNT(*) FROM expenses e {$where_sql}", $sp);

$sel_types  = $types . 'ii';
$sel_params = array_merge($params, [$pager['limit'], $pager['offset']]);

$rows = db_select(
    "SELECT e.*, c.company_name
     FROM expenses e
     LEFT JOIN clients c ON c.id = e.client_id
     {$where_sql}
     ORDER BY e.expense_date DESC, e.id DESC
     LIMIT ? OFFSET ?",
    array_merge([$sel_types], $sel_params)
);

/* Client list */
$client_list = db_select("SELECT id, company_name FROM clients WHERE is_active = 1 ORDER BY company_name ASC");

/* ══════════════════════════════════════════════════════════════
 *  RENDER
 * ══════════════════════════════════════════════════════════════ */
$page_title = 'Expenses';
require_once __DIR__ . '/header.php';
?>

<!-- ═══ Page Header ═════════════════════════════════════════════ -->
<div class="lp-page-header">
    <div>
        <h1><i class="bi bi-wallet2 me-2" style="font-size:1.4rem"></i>Expenses</h1>
        <div class="lp-subtitle">Track business expenses, vendor payments, and billable costs</div>
    </div>
    <button class="btn btn-lp-primary" data-bs-toggle="modal" data-bs-target="#expenseModal"
            onclick="resetExpenseForm()">
        <i class="bi bi-plus-lg me-1"></i> Add Expense
    </button>
</div>

<!-- ═══ Stats ═══════════════════════════════════════════════════ -->
<div class="row g-3 mb-4">
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon red"><i class="bi bi-cash-stack"></i></div>
            <div><div class="lp-stat-label">Total Expenses</div>
                 <div class="lp-stat-value"><?php echo format_currency($stat_total); ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon amber"><i class="bi bi-calendar-check"></i></div>
            <div><div class="lp-stat-label">This Month</div>
                 <div class="lp-stat-value"><?php echo format_currency($stat_month); ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon blue"><i class="bi bi-receipt-cutoff"></i></div>
            <div><div class="lp-stat-label">Billable</div>
                 <div class="lp-stat-value"><?php echo format_currency($stat_billable); ?></div></div>
        </div></div>
    </div>
    <div class="col-6 col-lg-3">
        <div class="lp-card h-100"><div class="lp-stat">
            <div class="lp-stat-icon purple"><i class="bi bi-hash"></i></div>
            <div><div class="lp-stat-label">Entries</div>
                 <div class="lp-stat-value"><?php echo $stat_count; ?></div></div>
        </div></div>
    </div>
</div>

<!-- ═══ Filters ═════════════════════════════════════════════════ -->
<div class="lp-card mb-4">
    <div class="card-body py-3">
        <form method="GET" action="expenses.php" class="row g-2 align-items-end">
            <div class="col-lg-2 col-md-4">
                <div class="input-group">
                    <span class="input-group-text bg-white border-end-0" style="border-color:var(--lp-silver)">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input type="text" name="search" class="form-control border-start-0 ps-0"
                           style="border-color:var(--lp-silver); font-size:.875rem"
                           placeholder="Search…" value="<?php echo h($search); ?>">
                </div>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <select name="category" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                    <option value="<?php echo h($cat); ?>" <?php echo $cat_filter === $cat ? 'selected' : ''; ?>>
                        <?php echo h($cat); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <select name="billable" class="form-select" style="border-color:var(--lp-silver); font-size:.875rem">
                    <option value="all" <?php echo $bill_filter === 'all' ? 'selected' : ''; ?>>All</option>
                    <option value="yes" <?php echo $bill_filter === 'yes' ? 'selected' : ''; ?>>Billable</option>
                    <option value="no"  <?php echo $bill_filter === 'no'  ? 'selected' : ''; ?>>Non-Billable</option>
                </select>
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_from" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_from); ?>" title="From">
            </div>
            <div class="col-lg-2 col-md-3 col-6">
                <input type="date" name="date_to" class="form-control"
                       style="border-color:var(--lp-silver); font-size:.835rem"
                       value="<?php echo h($date_to); ?>" title="To">
            </div>
            <div class="col-lg-2 col-md-6 d-flex gap-2">
                <button type="submit" class="btn btn-lp-primary"><i class="bi bi-funnel me-1"></i> Filter</button>
                <?php $hf = ($search !== '' || $cat_filter !== 'all' || $bill_filter !== 'all' || $date_from !== '' || $date_to !== '');
                if ($hf): ?>
                <a href="expenses.php" class="btn btn-lp-outline"><i class="bi bi-x-lg me-1"></i> Clear</a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- ═══ Expense List ════════════════════════════════════════════ -->
<?php if (empty($rows)): ?>
    <div class="lp-card"><div class="lp-empty">
        <i class="bi bi-wallet2 d-block"></i>
        <p><?php echo $hf ? 'No expenses match your filters.' : 'No expenses recorded yet.'; ?></p>
    </div></div>
<?php else: ?>

<!-- Desktop -->
<div class="lp-card mb-4 d-none d-md-block">
    <div class="lp-table-responsive">
        <table class="lp-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Category</th>
                    <th>Description</th>
                    <th>Vendor</th>
                    <th>Client</th>
                    <th class="text-center">Billable</th>
                    <th class="text-end">Amount</th>
                    <th class="text-end">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($rows as $r): ?>
                <tr>
                    <td style="font-size:.84rem"><?php echo date(DATE_FORMAT, strtotime($r['expense_date'])); ?></td>
                    <td>
                        <span class="badge rounded-pill"
                              style="background:var(--lp-red-soft, #fef2f2); color:var(--lp-red); font-size:.72rem; font-weight:500">
                            <?php echo h($r['category']); ?>
                        </span>
                    </td>
                    <td>
                        <?php echo h($r['description']); ?>
                        <?php if ($r['reference']): ?>
                        <small class="d-block text-muted" style="font-size:.72rem">Ref: <?php echo h($r['reference']); ?></small>
                        <?php endif; ?>
                    </td>
                    <td style="font-size:.84rem"><?php echo h($r['vendor'] ?: '—'); ?></td>
                    <td style="font-size:.84rem; color:var(--lp-slate-mid)">
                        <?php echo h($r['company_name'] ?? '—'); ?>
                    </td>
                    <td class="text-center">
                        <?php if ((int)$r['is_billable']): ?>
                            <span class="badge rounded-pill" style="background:var(--lp-blue-soft); color:var(--lp-blue); font-size:.68rem">Billable</span>
                        <?php else: ?>
                            <span class="text-muted" style="font-size:.78rem">—</span>
                        <?php endif; ?>
                    </td>
                    <td class="text-end fw-semibold" style="color:var(--lp-red)">
                        -<?php echo format_currency((float)$r['amount']); ?>
                    </td>
                    <td class="text-end">
                        <button class="btn btn-sm btn-lp-outline me-1" style="padding:.2rem .5rem; font-size:.78rem"
                                onclick="editExpense(<?php echo h(json_encode($r)); ?>)">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <form method="POST" action="expenses.php" class="d-inline"
                              data-confirm="Delete this expense?">
                            <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="expense_id" value="<?php echo (int)$r['id']; ?>">
                            <button type="submit" class="btn btn-sm text-danger border-0"
                                    style="padding:.2rem .5rem; font-size:.8rem"><i class="bi bi-trash"></i></button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Mobile -->
<div class="d-md-none mb-4">
    <?php foreach ($rows as $r): ?>
    <div class="lp-card mb-3"><div class="card-body">
        <div class="d-flex justify-content-between align-items-start mb-2">
            <div>
                <div class="fw-semibold" style="color:var(--lp-navy); font-size:.92rem"><?php echo h($r['description']); ?></div>
                <small class="text-muted"><?php echo h($r['vendor'] ?: $r['company_name'] ?? 'No vendor'); ?></small>
            </div>
            <span class="fw-bold" style="color:var(--lp-red)">-<?php echo format_currency((float)$r['amount']); ?></span>
        </div>
        <div class="d-flex gap-2 flex-wrap mb-2" style="font-size:.8rem">
            <span><i class="bi bi-calendar3 me-1 text-muted"></i><?php echo date(DATE_FORMAT, strtotime($r['expense_date'])); ?></span>
            <span class="badge rounded-pill" style="background:var(--lp-red-soft, #fef2f2); color:var(--lp-red); font-size:.7rem"><?php echo h($r['category']); ?></span>
            <?php if ((int)$r['is_billable']): ?>
            <span class="badge rounded-pill" style="background:var(--lp-blue-soft); color:var(--lp-blue); font-size:.7rem">Billable</span>
            <?php endif; ?>
        </div>
        <div class="d-flex gap-2 pt-2" style="border-top:1px solid var(--lp-silver)">
            <button class="btn btn-sm btn-lp-outline" style="font-size:.78rem"
                    onclick="editExpense(<?php echo h(json_encode($r)); ?>)">
                <i class="bi bi-pencil me-1"></i> Edit
            </button>
            <form method="POST" action="expenses.php" class="d-inline" data-confirm="Delete?">
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="expense_id" value="<?php echo (int)$r['id']; ?>">
                <button type="submit" class="btn btn-sm text-danger border-0" style="font-size:.78rem">
                    <i class="bi bi-trash me-1"></i> Delete
                </button>
            </form>
        </div>
    </div></div>
    <?php endforeach; ?>
</div>

<!-- Pagination -->
<?php if ($pager['total_pages'] > 1): ?>
<nav><ul class="pagination lp-pagination justify-content-center mb-0">
    <li class="page-item <?php echo !$pager['has_prev'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] - 1])); ?>"><i class="bi bi-chevron-left"></i></a></li>
    <?php
    $rng = 2; $s = max(1, $pager['current']-$rng); $e = min($pager['total_pages'], $pager['current']+$rng);
    if ($s > 1): ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>1])); ?>">1</a></li><?php if ($s > 2): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; endif;
    for ($p=$s; $p<=$e; $p++): ?><li class="page-item <?php echo $p===$pager['current']?'active':''; ?>"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$p])); ?>"><?php echo $p; ?></a></li><?php endfor;
    if ($e < $pager['total_pages']): if ($e < $pager['total_pages']-1): ?><li class="page-item disabled"><span class="page-link">…</span></li><?php endif; ?><li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page'=>$pager['total_pages']])); ?>"><?php echo $pager['total_pages']; ?></a></li><?php endif; ?>
    <li class="page-item <?php echo !$pager['has_next'] ? 'disabled' : ''; ?>">
        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $pager['current'] + 1])); ?>"><i class="bi bi-chevron-right"></i></a></li>
</ul></nav>
<?php endif; ?>

<?php endif; ?>

<!-- ═══ Create / Edit Modal ═════════════════════════════════════ -->
<div class="modal fade" id="expenseModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content" style="border:0; border-radius:var(--lp-radius-lg); overflow:hidden">
            <form method="POST" action="expenses.php" class="lp-form" novalidate>
                <input type="hidden" name="<?php echo CSRF_TOKEN_NAME; ?>" value="<?php echo csrf_token(); ?>">
                <input type="hidden" name="action" value="save">
                <input type="hidden" name="edit_id" id="exp_edit_id" value="0">

                <div class="modal-header" style="background:var(--lp-snow); border-bottom:1px solid var(--lp-silver)">
                    <h5 class="modal-title" id="expenseModalLabel">
                        <i class="bi bi-wallet2 me-1"></i> Add Expense
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Category <span class="text-danger">*</span></label>
                            <select class="form-select" name="category" id="exp_category" required>
                                <option value="">Select…</option>
                                <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo h($cat); ?>"><?php echo h($cat); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Vendor</label>
                            <input type="text" class="form-control" name="vendor" id="exp_vendor"
                                   maxlength="150" placeholder="e.g. Amazon Web Services">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="description" id="exp_description"
                                   required maxlength="255" placeholder="What was this expense for?">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Amount <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><?php echo CURRENCY_SYMBOL; ?></span>
                                <input type="number" class="form-control" name="amount" id="exp_amount"
                                       step="0.01" min="0.01" required placeholder="0.00">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Date <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" name="expense_date" id="exp_date"
                                   value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Reference</label>
                            <input type="text" class="form-control" name="reference" id="exp_reference"
                                   maxlength="100" placeholder="e.g. RCT-9021">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Client <small class="text-muted">(optional)</small></label>
                            <select class="form-select" name="client_id" id="exp_client_id">
                                <option value="0">— None —</option>
                                <?php foreach ($client_list as $cl): ?>
                                <option value="<?php echo (int)$cl['id']; ?>"><?php echo h($cl['company_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6 d-flex align-items-end pb-1">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" role="switch"
                                       id="exp_billable" name="is_billable" value="1"
                                       style="width:2.5em; height:1.25em; cursor:pointer">
                                <label class="form-check-label ms-1" for="exp_billable"
                                       style="font-weight:500; cursor:pointer; font-size:.88rem">
                                    Billable to client
                                </label>
                            </div>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" id="exp_notes" rows="2"
                                      style="resize:vertical" placeholder="Optional notes…"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer" style="background:var(--lp-snow); border-top:1px solid var(--lp-silver)">
                    <button type="button" class="btn btn-lp-outline" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-lp-primary">
                        <i class="bi bi-check-lg me-1"></i> <span id="expSaveLabel">Save</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function resetExpenseForm() {
    document.getElementById('exp_edit_id').value     = '0';
    document.getElementById('exp_category').value    = '';
    document.getElementById('exp_vendor').value      = '';
    document.getElementById('exp_description').value = '';
    document.getElementById('exp_amount').value      = '';
    document.getElementById('exp_date').value        = '<?php echo date('Y-m-d'); ?>';
    document.getElementById('exp_reference').value   = '';
    document.getElementById('exp_client_id').value   = '0';
    document.getElementById('exp_billable').checked  = false;
    document.getElementById('exp_notes').value       = '';
    document.getElementById('expenseModalLabel').innerHTML = '<i class="bi bi-wallet2 me-1"></i> Add Expense';
    document.getElementById('expSaveLabel').textContent    = 'Save';
}

function editExpense(data) {
    document.getElementById('exp_edit_id').value     = data.id;
    document.getElementById('exp_category').value    = data.category;
    document.getElementById('exp_vendor').value      = data.vendor || '';
    document.getElementById('exp_description').value = data.description;
    document.getElementById('exp_amount').value      = parseFloat(data.amount).toFixed(2);
    document.getElementById('exp_date').value        = data.expense_date;
    document.getElementById('exp_reference').value   = data.reference || '';
    document.getElementById('exp_client_id').value   = data.client_id || '0';
    document.getElementById('exp_billable').checked  = parseInt(data.is_billable) === 1;
    document.getElementById('exp_notes').value       = data.notes || '';
    document.getElementById('expenseModalLabel').innerHTML = '<i class="bi bi-pencil me-1"></i> Edit Expense';
    document.getElementById('expSaveLabel').textContent    = 'Update';
    new bootstrap.Modal(document.getElementById('expenseModal')).show();
}
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
