<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  db.php — MySQLi Database Connection & Query Helpers
 * ═══════════════════════════════════════════════════════════════
 *
 *  This file provides:
 *    1. A singleton MySQLi connection (no duplicate connections)
 *    2. Prepared-statement wrappers for safe queries
 *    3. Transaction helpers (begin / commit / rollback)
 *    4. Automatic schema installation on first run
 *
 *  Usage:
 *    require_once 'db.php';
 *    $rows = db_select("SELECT * FROM users WHERE role = ?", ['s', 'accountant']);
 *
 *  SECURITY: Every query that touches user input MUST go through
 *  the prepared-statement helpers below. Raw concatenation is
 *  forbidden throughout the entire application.
 * ═══════════════════════════════════════════════════════════════
 */

require_once __DIR__ . '/config.php';

/* ──────────────────────────────────────────────────────────────
 *  SINGLETON CONNECTION
 *  Returns the same mysqli instance on every call.
 * ─────────────────────────────────────────────────────────────── */

function db_connect(): mysqli
{
    static $conn = null;

    if ($conn !== null && $conn->ping()) {
        return $conn;
    }

    /* Enable exception-based error reporting for mysqli */
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

    try {
        $conn = new mysqli(
            DB_HOST,
            DB_USER,
            DB_PASS,
            DB_NAME,
            DB_PORT
        );

        /* Force UTF-8 on the connection */
        $conn->set_charset(DB_CHARSET);

        /* Strict SQL mode for data integrity */
        $conn->query("SET sql_mode = 'STRICT_TRANS_TABLES,ERROR_FOR_DIVISION_BY_ZERO,NO_ENGINE_SUBSTITUTION'");

    } catch (mysqli_sql_exception $e) {
        /*
         * If the database doesn't exist yet, create it and run schema.
         * This makes first-time setup completely automatic.
         */
        if ($e->getCode() === 1049) { // ER_BAD_DB_ERROR
            return db_install_schema();
        }
        error_log('DB Connection Error: ' . $e->getMessage());
        die('<h1>Database Error</h1><p>Could not connect. Check config.php credentials.</p>');
    }

    return $conn;
}


/* ──────────────────────────────────────────────────────────────
 *  AUTOMATIC SCHEMA INSTALLER
 *  Runs schema.sql when the database does not yet exist.
 * ─────────────────────────────────────────────────────────────── */

function db_install_schema(): mysqli
{
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

    /* Connect without selecting a database */
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, '', DB_PORT);
    $conn->set_charset(DB_CHARSET);

    $schemaFile = __DIR__ . '/schema.sql';

    if (!file_exists($schemaFile)) {
        die('<h1>Setup Error</h1><p>schema.sql not found. Place it in the application root.</p>');
    }

    /* Read and execute the full schema (multi_query for multiple statements) */
    $sql = file_get_contents($schemaFile);
    if (!$conn->multi_query($sql)) {
        die('<h1>Schema Error</h1><p>' . htmlspecialchars($conn->error) . '</p>');
    }

    /* Flush all result sets produced by multi_query */
    do {
        if ($result = $conn->store_result()) {
            $result->free();
        }
    } while ($conn->next_result());

    /* Close and re-connect to the newly created database */
    $conn->close();

    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME, DB_PORT);
    $conn->set_charset(DB_CHARSET);
    $conn->query("SET sql_mode = 'STRICT_TRANS_TABLES,ERROR_FOR_DIVISION_BY_ZERO,NO_ENGINE_SUBSTITUTION'");

    return $conn;
}


/* ──────────────────────────────────────────────────────────────
 *  PREPARED-STATEMENT HELPERS
 * ──────────────────────────────────────────────────────────────
 *
 *  $types_and_params is a flat array where the FIRST element is
 *  a mysqli type string ("ssi", "sd", etc.) and the remaining
 *  elements are the bound values, matching that type string.
 *
 *  Example:
 *    db_select(
 *        "SELECT * FROM invoices WHERE client_id = ? AND status = ?",
 *        ['is', 5, 'paid']
 *    );
 *
 *  Pass an empty array [] for queries with no parameters.
 * ─────────────────────────────────────────────────────────────── */

/**
 * Run a SELECT query and return all rows as associative arrays.
 *
 * @param  string $sql              Parameterized SQL
 * @param  array  $types_and_params [type_string, ...values] or []
 * @return array                    Array of associative-array rows
 */
function db_select(string $sql, array $types_and_params = []): array
{
    $conn = db_connect();
    $stmt = $conn->prepare($sql);

    if (!empty($types_and_params)) {
        $types  = array_shift($types_and_params);
        $stmt->bind_param($types, ...$types_and_params);
    }

    $stmt->execute();
    $result = $stmt->get_result();
    $rows   = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    return $rows;
}

/**
 * Run a SELECT query and return exactly one row (or null).
 *
 * @param  string     $sql
 * @param  array      $types_and_params
 * @return array|null Single associative-array row or null
 */
function db_select_one(string $sql, array $types_and_params = []): ?array
{
    $rows = db_select($sql, $types_and_params);
    return $rows[0] ?? null;
}

/**
 * Run a single scalar query (COUNT, SUM, MAX, etc.) and return the value.
 *
 * @param  string $sql
 * @param  array  $types_and_params
 * @return mixed  The first column of the first row, or null
 */
function db_scalar(string $sql, array $types_and_params = [])
{
    $conn = db_connect();
    $stmt = $conn->prepare($sql);

    if (!empty($types_and_params)) {
        $types = array_shift($types_and_params);
        $stmt->bind_param($types, ...$types_and_params);
    }

    $stmt->execute();
    $result = $stmt->get_result();
    $row    = $result->fetch_row();
    $stmt->close();

    return $row[0] ?? null;
}

/**
 * Run an INSERT, UPDATE, or DELETE statement.
 *
 * @param  string $sql
 * @param  array  $types_and_params
 * @return int    Number of affected rows
 */
function db_execute(string $sql, array $types_and_params = []): int
{
    $conn = db_connect();
    $stmt = $conn->prepare($sql);

    if (!empty($types_and_params)) {
        $types = array_shift($types_and_params);
        $stmt->bind_param($types, ...$types_and_params);
    }

    $stmt->execute();
    $affected = $stmt->affected_rows;
    $stmt->close();

    return $affected;
}

/**
 * Run an INSERT and return the auto-increment ID of the new row.
 *
 * @param  string $sql
 * @param  array  $types_and_params
 * @return int    The new row's ID
 */
function db_insert(string $sql, array $types_and_params = []): int
{
    $conn = db_connect();
    $stmt = $conn->prepare($sql);

    if (!empty($types_and_params)) {
        $types = array_shift($types_and_params);
        $stmt->bind_param($types, ...$types_and_params);
    }

    $stmt->execute();
    $id = (int) $conn->insert_id;
    $stmt->close();

    return $id;
}


/* ──────────────────────────────────────────────────────────────
 *  TRANSACTION HELPERS
 *  Wrap multi-step operations so they either all succeed or
 *  all fail. Critical for financial data integrity.
 *
 *  Usage:
 *    db_begin();
 *    try {
 *        db_insert("INSERT INTO ...", [...]);
 *        db_execute("UPDATE ...", [...]);
 *        db_commit();
 *    } catch (Exception $e) {
 *        db_rollback();
 *        throw $e;
 *    }
 * ─────────────────────────────────────────────────────────────── */

function db_begin(): void
{
    db_connect()->begin_transaction();
}

function db_commit(): void
{
    db_connect()->commit();
}

function db_rollback(): void
{
    db_connect()->rollback();
}


/* ──────────────────────────────────────────────────────────────
 *  PAGINATION HELPER
 *  Returns an array with offset, limit, total pages, etc.
 *
 *  Usage:
 *    $pager = db_paginate("SELECT COUNT(*) FROM invoices WHERE status = ?", ['s','paid']);
 *    $rows  = db_select("SELECT * FROM invoices WHERE status = ? LIMIT ? OFFSET ?",
 *                        ['sii', 'paid', $pager['limit'], $pager['offset']]);
 * ─────────────────────────────────────────────────────────────── */

function db_paginate(string $count_sql, array $types_and_params = [], int $per_page = ROWS_PER_PAGE): array
{
    $total_rows  = (int) db_scalar($count_sql, $types_and_params);
    $total_pages = max(1, (int) ceil($total_rows / $per_page));
    $current     = max(1, (int) ($_GET['page'] ?? 1));
    $current     = min($current, $total_pages);
    $offset      = ($current - 1) * $per_page;

    return [
        'total_rows'  => $total_rows,
        'total_pages' => $total_pages,
        'current'     => $current,
        'per_page'    => $per_page,
        'limit'       => $per_page,
        'offset'      => $offset,
        'has_prev'    => $current > 1,
        'has_next'    => $current < $total_pages,
    ];
}


/* ──────────────────────────────────────────────────────────────
 *  UTILITY: Safely get the last insert ID outside of db_insert
 * ─────────────────────────────────────────────────────────────── */

function db_last_id(): int
{
    return (int) db_connect()->insert_id;
}

/**
 * Escape a LIKE pattern so that user input is treated literally.
 * Pair with a prepared statement: WHERE col LIKE CONCAT('%', ?, '%')
 */
function db_escape_like(string $input): string
{
    return addcslashes($input, '%_\\');
}
