<?php
/**
 * ═══════════════════════════════════════════════════════════════
 *  LedgerPro — Professional Accounting Platform
 *  config.php — Core Application Configuration
 * ═══════════════════════════════════════════════════════════════
 *
 *  Central configuration hub. All database credentials, app
 *  constants, session settings, and environment flags live here.
 *  Include this file FIRST in every entry point.
 */

/* ── Error Reporting (disable display in production) ──────── */
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

/* ── Session Hardening ────────────────────────────────────── */
ini_set('session.cookie_httponly', 1);   // JS cannot read cookie
ini_set('session.cookie_secure', 0);     // Set 1 in production w/ HTTPS
ini_set('session.use_strict_mode', 1);   // Reject uninitialized session IDs
ini_set('session.cookie_samesite', 'Strict');

/* ── Database Credentials ─────────────────────────────────── */
define('DB_HOST',    'localhost');
define('DB_USER',    'root');
define('DB_PASS',    '');               // Set a strong password in production
define('DB_NAME',    'ledgerpro');
define('DB_PORT',    3306);
define('DB_CHARSET', 'utf8mb4');

/* ── Application Constants ────────────────────────────────── */
define('APP_NAME',    'LedgerPro');
define('APP_VERSION', '1.0.0');
define('APP_URL',     'http://localhost/accounting-app');  // No trailing slash

/* ── User Role Constants ──────────────────────────────────── */
define('ROLE_ACCOUNTANT', 'accountant');
define('ROLE_CLIENT',     'client');

/* ── Invoice Status Constants ─────────────────────────────── */
define('STATUS_DRAFT',    'draft');
define('STATUS_SENT',     'sent');
define('STATUS_PAID',     'paid');
define('STATUS_OVERDUE',  'overdue');
define('STATUS_CANCELLED','cancelled');

/* ── Expense Category Defaults ────────────────────────────── */
define('EXPENSE_CATEGORIES', json_encode([
    'Office Supplies',
    'Travel',
    'Meals & Entertainment',
    'Utilities',
    'Rent',
    'Insurance',
    'Professional Services',
    'Marketing',
    'Software & Subscriptions',
    'Equipment',
    'Payroll',
    'Taxes',
    'Miscellaneous'
]));

/* ── Currency & Locale ────────────────────────────────────── */
define('CURRENCY_SYMBOL', '$');
define('CURRENCY_CODE',   'USD');
define('DATE_FORMAT',     'M d, Y');          // Display format
define('DATE_FORMAT_DB',  'Y-m-d');           // Storage format
define('DATETIME_FORMAT', 'M d, Y h:i A');

/* ── Pagination ───────────────────────────────────────────── */
define('ROWS_PER_PAGE', 15);

/* ── File Upload Limits ───────────────────────────────────── */
define('MAX_UPLOAD_SIZE',  5 * 1024 * 1024);  // 5 MB
define('ALLOWED_EXTENSIONS', ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx', 'xls', 'xlsx']);

/* ── Security ─────────────────────────────────────────────── */
define('CSRF_TOKEN_NAME', 'csrf_token');
define('PASSWORD_MIN_LEN', 8);

/**
 * Generate or retrieve the current CSRF token.
 * Must be called AFTER session_start().
 */
function csrf_token(): string
{
    if (empty($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Validate a submitted CSRF token against the session token.
 */
function csrf_validate(string $token): bool
{
    return isset($_SESSION[CSRF_TOKEN_NAME])
        && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Regenerate the CSRF token (call after successful form submission).
 */
function csrf_regenerate(): void
{
    $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
}

/**
 * Format a number as currency for display.
 */
function format_currency(float $amount): string
{
    return CURRENCY_SYMBOL . number_format($amount, 2);
}

/**
 * Sanitize a string for safe HTML output.
 */
function h(string $str): string
{
    return htmlspecialchars($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

/**
 * Redirect to an internal URL and terminate.
 */
function redirect(string $path): void
{
    header('Location: ' . APP_URL . '/' . ltrim($path, '/'));
    exit;
}

/**
 * Flash message helper — set or get session flash messages.
 */
function flash(string $key, ?string $message = null): ?string
{
    if ($message !== null) {
        $_SESSION['flash'][$key] = $message;
        return null;
    }
    $msg = $_SESSION['flash'][$key] ?? null;
    unset($_SESSION['flash'][$key]);
    return $msg;
}

/**
 * Check if the current user is logged in.
 */
function is_logged_in(): bool
{
    return !empty($_SESSION['user_id']);
}

/**
 * Check if the current user has the accountant role.
 */
function is_accountant(): bool
{
    return is_logged_in() && ($_SESSION['user_role'] ?? '') === ROLE_ACCOUNTANT;
}

/**
 * Check if the current user has the client role.
 */
function is_client(): bool
{
    return is_logged_in() && ($_SESSION['user_role'] ?? '') === ROLE_CLIENT;
}

/**
 * Require the user to be logged in; redirect to login otherwise.
 */
function require_login(): void
{
    if (!is_logged_in()) {
        flash('error', 'Please log in to continue.');
        redirect('login.php');
    }
}

/**
 * Require the accountant role; show 403 otherwise.
 */
function require_accountant(): void
{
    require_login();
    if (!is_accountant()) {
        http_response_code(403);
        die('<h1>403 — Access Denied</h1><p>Accountant privileges required.</p>');
    }
}
